/**
 * Copyright (C) 2008-2013 LimeTri. All rights reserved.
 *
 * AgroSense is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * There are special exceptions to the terms and conditions of the GPLv3 as it is applied to
 * this software, see the FLOSS License Exception
 * <http://www.agrosense.eu/foss-exception.html>.
 *
 * AgroSense is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AgroSense.  If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.util.api.geo;

import java.awt.Image;
import java.beans.PropertyChangeListener;

import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.Point;
import nl.bebr.util.api.geo.aspect.HasBoundingBox;

/**
 * Defines geographical capabilities for an object. Geographical objects can be displayed on a map.
 *
 * @author Timon Veenstra
 * @param <R> type of object to render
 */
public interface Geographical<R> extends HasBoundingBox {

    public static final String PROP_RENDER_OBJECT = "renderObject";

    /**
     * Get type class type of the wrapped object
     *
     * @return
     */
    Class<R> getType();

    /**
     * Get the centroid (center of weight) for this geographical object.
     *
     * @return
     * @see Geometry#getCentroid()
     */
    Point getCentroid();

    @Override
    Geometry getBoundingBox();

    /**
     * Returns an object that is to be rendered on the map/layer valid within the envelope. Can return null if nothing should be
     * rendered within the envelope.
     *
     * @param envelope
     * @return
     */
    R getRenderObject(Envelope envelope);

    /**
     * Get tool tip text. This can for example be the ID of the wrapped object followed by its name.
     *
     * @return TooltipText as String
     */
    String getTooltipText();

    /**
     * Get an image to be drawn on top of the render object. Can be null if no icon should be drawn.
     *
     * @return The icon image, or null if no image should be drawn.
     */
    //FIXME: make icons usable with both swing and javafx.
    //Can't use a resource path/URL instead because current implementations 
    //return dynamically generated images (e.g. base icon merged with a badge).
    //javafx.embed.swing.SwingFXUtils can convert between the two image types (but requires awt BufferedImage as input).
    Image getIcon();

    /**
     * Get the text for a label to be drawn below the icon.
     *
     * @return The label text, or null if no label should be drawn.
     * @see #getIcon()
     */
    String getIconLabel();

    /**
     * Adds a listener for PropertyChangeEvents to this geographical.
     *
     * @param listener The PropertyChangeListener to add
     */
    void addPropertyChangeListener(PropertyChangeListener listener);

    /**
     * Removes the given PropertyChangeListener from this geographical.
     *
     * @param listener The listener to remove
     */
    void removePropertyChangeListener(PropertyChangeListener listener);

    // using separate nested interface for now so we don't have to update all Geographical implementations.
    interface AsyncRenderObject {

        /**
         * Indicates that the object returned by {@code getRenderObject()} will be outdated.
         * <p>
         * Either a new object is currently being generated, or generation will be triggered by the call to
         * {@code getRenderObject()}. This information can be used by views (map widgets, preview panels) to render e.g. an
         * hourglass icon or a progress handle on to of the outdated object.</p>
         *
         * @return
         */
        boolean hasStaleRenderObject();
    }

//    /**
//     * Static helper class for merging specific Geographical actions with the generic Geographical actions
//     */
//    public class Actions {
//        private static final Logger LOG = Logger.getLogger("LayerObject.Actions");
//
//        private Actions(){}
//        
//        /**
//         * Merge the layerobject specific actions with the global layerobject actions
//         *
//         * @param layerObject
//         * @return
//         */
//        public static Action[] mergeActions(Geometrical layerObject) {
//            return mergeActions(layerObject, (Action[]) null);
//        }
//
//        /**
//         * Merge the layerobject specific actions with the global layerobject actions
//         *
//         * @param layerObject
//         * @return
//         */
//        public static Action[] mergeActions(Node node, Action... actions) {
//            // first add all global layer actions from the lookup
//            List<Action> globalActions = new ArrayList<Action>();
//           //
//            // add layer specific actions
//            //
//            GeographicalAction[] specificActions = layerObject.getActions();
//            if (specificActions != null && specificActions.length > 0) {
//                globalActions.addAll(Arrays.asList(specificActions));
//            }
//
//            if (actions != null && actions.length > 0) {
//                globalActions.addAll(Arrays.asList(actions));
//            }
//            return globalActions.toArray(new Action[]{});
//        }
//    }
}
