/**
 * Copyright (C) 2008-2013 LimeTri. All rights reserved.
 *
 * AgroSense is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * There are special exceptions to the terms and conditions of the GPLv3 as it
 * is applied to this software, see the FLOSS License Exception
 * <http://www.agrosense.eu/foss-exception.html>.
 *
 * AgroSense is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * AgroSense. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.util.api.geo.util;

import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryFactory;
import com.vividsolutions.jts.geom.LinearRing;
import com.vividsolutions.jts.geom.Point;
import com.vividsolutions.jts.geom.Polygon;
import com.vividsolutions.jts.io.ParseException;
import com.vividsolutions.jts.io.WKTReader;
import java.awt.geom.Point2D;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geotools.factory.GeoTools;
import org.geotools.geometry.jts.JTS;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.geotools.referencing.CRS;
import org.geotools.referencing.GeodeticCalculator;
import org.geotools.referencing.ReferencingFactoryFinder;
import org.opengis.geometry.MismatchedDimensionException;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.operation.CoordinateOperation;

/**
 *
 * @author Gerben Feenstra
 */
public class GeometryTools {

    private static GeometryFactory geometryFactory;
    private static final double ACREAGE_DEVIDER = 10000d;
    private static final Logger LOGGER = Logger.getLogger("eu.agrosense.client.lib.geotools");
    public static final int GEOMETRY_INVALID_CUT_EDGES = 1, GEOMETRY_INVALID_DANGLES = 2, GEOMETRY_INVALID_RING_LINES = 3;

    /*
     * Uses JTSFactoryFinder to find the GeometryFactory
     */
    private static GeometryFactory getGeometryFactory() {
        if (geometryFactory != null) {
            return geometryFactory;
        } else {
            return (geometryFactory = JTSFactoryFinder.getGeometryFactory(null));
        }
    }

    /**
     * Returns a geometry made from the specified WKT String
     *
     * @param wkt The WKT String
     * @return The Geometry extracted from the String
     */
    public static Geometry wktToGeometry(String wkt) {
        try {

            WKTReader wKTReader = new WKTReader(getGeometryFactory());
            return wKTReader.read(wkt);
        } catch (ParseException ex) {
            throw new IllegalArgumentException("Provided argumment " + wkt + " did not contain a parseable geometry ", ex);
        }
    }

    /**
     * Create a Point from a longitude and a latitude value using the specified
     * geometryFactory
     *
     * @param geometryFactory The geometryFactory to use
     * @param longitude
     * @param latitude
     * @return Point created from the specified coordinates
     */
    public static Point getPoint(GeometryFactory geometryFactory, double longitude, double latitude) {
        Coordinate coordinate = new Coordinate(longitude, latitude);
        Point geometry = geometryFactory.createPoint(coordinate);
        return geometry;
    }

    /**
     * Convenience method, retrieves a GeometryFactory from the JTSFactoryFinder
     * using getGeometryFactory and calls getPoint(GeometryFactory
     * geometryFactory, double longitude, double latitude)
     *
     * @param longitude
     * @param latitude
     * @return Point created from the specified coordinates
     */
    public static Point getPoint(double longitude, double latitude) {
        return getPoint(getGeometryFactory(), longitude, latitude);
    }

    /**
     * Return acreage in Hectares
     *
     * @param acreage as provided by vivid solutions Shape
     * @return Acreage in HA
     */
    public static Double getAcreageInHA(Double acreage) {
        return acreage > 0 ? acreage / ACREAGE_DEVIDER : acreage;
    }

    /**
     * Get the acreage from geometry in format for a particular country. As of
     * now it returns the geometry formated in EPSG:28992 or Amersfoort / RD
     * New. TODO: make this method generic and work with some sort of properties
     * file which will give back the appropriate country format
     * http://www.geotoolkit.org/modules/referencing/supported-codes.html
     *
     * @param geometry the geometry to format to the new country format
     * @return the acreage of the formatted geometry.
     * @Throws UnsupportedOperationException when the param geometry is not in
     * the Netherlands
     * @deprecated use use {@link GeometryTools#getAreaInHA}
     */
    @Deprecated
    public static Double getAcreageFromGeometry(Geometry geometry) {
        Geometry transFormed = null;
        Double faultValue = new Double(0.0);
        // make geometry for the netherlands
        String netherlandsWKT = "MULTIPOLYGON (((3.10000 51.20000, 3.10000 53.55000, 7.210000 53.55000, 7.21000 50.70000, 3.10000 51.20000)))";
        Geometry netherLands = wktToGeometry(netherlandsWKT);
        if (netherLands.contains(geometry)) {
            try {
                CoordinateReferenceSystem sourceReference = CRS.decode("CRS:84");
                CoordinateReferenceSystem targetReference = CRS.decode("EPSG:28992");

                CoordinateOperation coFactory = ReferencingFactoryFinder.getCoordinateOperationFactory(
                        GeoTools.getDefaultHints()).createOperation(sourceReference, targetReference);

                transFormed = JTS.transform(geometry, coFactory.getMathTransform());
                return transFormed.getArea();
            } catch (org.opengis.referencing.operation.TransformException | MismatchedDimensionException | FactoryException ex) {
                return faultValue;
            }

        }
        return faultValue;
    }

    /**
     * <p>Get the acreage from a geometry Will search the UTMZone for this
     * geometry and calculate the acreage based on that</p>
     *
     * <p>The precision has a 0.2 percent deviation based on calculation with
     * vicinity calculation and comparison with quantum Gis</p> <p> In the
     * return value this deviation is accounted for If for instance the
     * calculated value is 189.2026514 the digits after 189.2 are insignificant
     * therefore only 189.2 is returned If the calculated value is
     * 15896141556.214451 the insignificance occurs after the first four numbers
     * 1589 therefore the rest of the numbers are filled with zeros the return
     * value will be 158900000000</p>
     *
     * @param geometry the geometry to calculate the acreage for
     * @return the calculated value or 0.0 if an error occurs
     */
    public static Double getAreaInHA(Geometry geometry) {
        final double ACREAGE_DIVIDER = 10000;
        Geometry transFormed;
        Double faultValue = new Double(0.0);
        UTMZone uTMZone = UTMZone.fromGPSCoordinates(geometry.getCentroid().getY(), geometry.getCentroid().getX());
        try {
            CoordinateReferenceSystem sourceReference = CRS.decode("CRS:84");
            CoordinateReferenceSystem targetReference = CRS.decode(uTMZone.getEPSG());

            CoordinateOperation coFactory = ReferencingFactoryFinder.getCoordinateOperationFactory(
                    GeoTools.getDefaultHints()).createOperation(sourceReference, targetReference);

            transFormed = JTS.transform(geometry, coFactory.getMathTransform());
            Double calculatedArea = transFormed.getArea();
            calculatedArea = (calculatedArea / ACREAGE_DIVIDER);
            return getPrecisionDouble(calculatedArea);
        } catch (org.opengis.referencing.operation.TransformException | MismatchedDimensionException | FactoryException ex) {
            LOGGER.log(Level.SEVERE, "Unable to transform geometry {0} to UTMZone {1} , Exception occurred: {2}", new Object[]{geometry, uTMZone.getEPSG(), ex.getMessage()});
            return faultValue;
        }
    }

    /**
     * Calculates a double based on the input The precision is a thousand so the
     * first four numbers are precise the rest becomes 0's 111.111 will result
     * in 111.1 111111.111 will result in 111100
     *
     * @param toTransform
     * @return the transformed number
     */
    protected static double getPrecisionDouble(Double toTransform) {
        final int SCALE = 0;
        final int BIGDECIMAL_BASE_POWER_OF_TEN = 1;
        int precision = 4;
        double log10 = Math.ceil(Math.log10(toTransform));
        BigDecimal transForm = BigDecimal.valueOf(toTransform);
        BigDecimal factor = BigDecimal.valueOf(BIGDECIMAL_BASE_POWER_OF_TEN).scaleByPowerOfTen((int) (precision - log10));
        return transForm.multiply(factor).setScale(SCALE, RoundingMode.FLOOR).divide(factor).doubleValue();
    }

    /**
     * Calculates destination point given start point lat/long, bearing &
     * distance, using GeodeticCalculator
     *
     * Uses WGS84 elipsoid
     *
     * @param lat1 latitude first point in decimal degrees
     * @param lon1 longitude first point in decimal degrees
     * @param brng initial bearing in decimal degrees from due north ranging
     * from -180 to 180 where 0 is due north
     * @param dist distance along bearing in metres @returns the coordinate
     * calculated
     *
     */
    public static Point2D getDestinationPoint(double lat1, double lon1, double brng, double dist) {
        GeodeticCalculator calculator = new GeodeticCalculator();
        calculator.setStartingGeographicPoint(lon1, lat1);
        calculator.setDirection(brng, dist);
        return calculator.getDestinationGeographicPoint();
    }

    /**
     * Calculates distance between supplied starting latitude longitude and
     * supplied end latitude longitude
     *
     * Uses WGS84 elipsoid
     *
     * @param lat1 latitude first point in decimal degrees
     * @param lon1 longitude first point in decimal degrees
     * @param lat2 latitude second point in decimal degrees
     * @param lon2 longitude second point in decimal degrees @returns the
     * distance calculated
     *
     */
    public static double getDistance(double lat1, double lon1, double lat2, double lon2) {
        GeodeticCalculator calculator = new GeodeticCalculator();
        calculator.setStartingGeographicPoint(lon1, lat1);
        calculator.setDestinationGeographicPoint(lon2, lat2);
        return calculator.getOrthodromicDistance();
    }

    /**
     * Calculates initial bearing between supplied starting latitude longitude
     * and supplied end latitude longitude.
     *
     * Take notice!! initial bearing the bearing over long distances is a great
     * circle path so the midpoint and final bearing can vary a lot
     *
     * Uses WGS84 elipsoid
     *
     * @param lat1 latitude first point in decimal degrees
     * @param lon1 longitude first point in decimal degrees
     * @param lat2 latitude second point in decimal degrees
     * @param lon2 longitude second point in decimal degrees @returns the
     * bearing calculated in decimal degrees from due north ranging from -180 to
     * 180 where 0 is due north
     *
     */
    public static double getBearing(double lat1, double lon1, double lat2, double lon2) {
        GeodeticCalculator calculator = new GeodeticCalculator();
        calculator.setStartingGeographicPoint(lon1, lat1);
        calculator.setDestinationGeographicPoint(lon2, lat2);
        return calculator.getAzimuth();
    }
    
    /**
     * Creates an axis-aligned rectangular polygon.
     * 
     * E.g. for an envelope or a grid cell.
     * 
     * @param minX
     * @param minY
     * @param maxX
     * @param maxY
     * @param factory
     * @return 
     */
    public static Polygon createRectangle(double minX, double minY, double maxX, double maxY, GeometryFactory factory) {
        assert minX < maxX;
        assert minY < maxY;
        assert factory != null;

        Coordinate[] coordinates = new Coordinate[5];
        coordinates[0] = new Coordinate(minX, minY);
        coordinates[1] = new Coordinate(maxX, minY);
        coordinates[2] = new Coordinate(maxX, maxY);
        coordinates[3] = new Coordinate(minX, maxY);
        coordinates[4] = new Coordinate(coordinates[0]); // close ring

        LinearRing ring = factory.createLinearRing(coordinates);
        Polygon polygon = factory.createPolygon(ring, null);

        return polygon;
    }         

}
