/**
 * Copyright (C) 2008-2013 LimeTri. All rights reserved.
 *
 * AgroSense is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * There are special exceptions to the terms and conditions of the GPLv3 as it
 * is applied to this software, see the FLOSS License Exception
 * <http://www.agrosense.eu/foss-exception.html>.
 *
 * AgroSense is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * AgroSense. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.util.api.geo.util;

/**
 * 
 * Represents an UTMZone.
 * <p>
 * UTM is a world covering projection, where each hemisphere is divided into
 * 60 parts. This provides a global working projection method for area 
 * calculations.
 * </p><p>
 * To retrieve the UTMZone for a geometry, use {@link UTMZone#fromGPSCoordinates(double, double) }
 * and provide the coordinate from the center point. Even if the geometry
 * would be on the border of 2 zones, both zones would be just as accurate, 
 * so the center point is a valid option.
 * </p>
 *
 * @author Timon Veenstra <monezz@gmail.com>
 */
public class UTMZone {
    public enum Hemisphere{
        NORTHERN, SOUTHERN
    }
    
    private final Hemisphere hemisphere;
    private final int number;
    private final String EPSG;

    public UTMZone(int number, Hemisphere hemisphere) {
        this.hemisphere = hemisphere;
        this.number = number;
        if (number<0 || number>60){
            throw new IllegalArgumentException("Argument number should be between 0 and 60");
        }
        String no = String.valueOf(number);
        no = (no.length()==1)?"0"+no:no;
        
        EPSG = "EPSG:32"+(Hemisphere.NORTHERN.equals(hemisphere)?"6":"7")+no;
    }

    public String getEPSG() {
        return EPSG;
    }

    public Hemisphere getHemisphere() {
        return hemisphere;
    }

    public int getNumber() {
        return number;
    }
    
    /**
     * Determine the UTMZone for given latitude and longitude.
     * 
     * @param latitude
     * @param longitude
     * @return 
     */
    public static UTMZone fromGPSCoordinates(double latitude, double longitude){
        return new UTMZone(longitudeToZoneNumber(longitude), latitudeToHemisphere(latitude));
    }
    
    public static int longitudeToZoneNumber(double longitude){
        return new Double(Math.floor((longitude+180.0)/6.0)).intValue()+1;
    }
    
    public static Hemisphere latitudeToHemisphere(double latitude){
        return (latitude >= 0)?Hemisphere.NORTHERN:Hemisphere.SOUTHERN;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final UTMZone other = (UTMZone) obj;
        if (this.hemisphere != other.hemisphere) {
            return false;
        }
        if (this.number != other.number) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 61 * hash + (this.hemisphere != null ? this.hemisphere.hashCode() : 0);
        hash = 61 * hash + this.number;
        return hash;
    }
    
    
    
}
