/**
 * Copyright (C) 2008-2013 LimeTri. All rights reserved.
 *
 * AgroSense is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * There are special exceptions to the terms and conditions of the GPLv3 as it
 * is applied to this software, see the FLOSS License Exception
 * <http://www.agrosense.eu/foss-exception.html>.
 *
 * AgroSense is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * AgroSense. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.util.api.model.id;

import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Objects;

/**
 *
 * @author johan
 */
public interface CompositeId {
    
    String[] getParts();
    
    abstract class BaseImpl implements CompositeId, Serializable {
        protected final String identifier;
        protected transient String[] parts;

        public BaseImpl(String identifier) {
            this.identifier = Objects.requireNonNull(identifier);
            init();
        }
        
        private void init() {
            initParts();
            validate();
        }
        
        protected abstract void initParts();
        
        protected abstract void validate();
        
        @Override
        public String[] getParts() {
            return Arrays.copyOf(parts, parts.length);
        }
        
        @Override
        public String toString() {
            return identifier;
        }

        @Override
        public int hashCode() {
            int hash = 5;
            hash = 71 * hash + Objects.hashCode(this.identifier);
            return hash;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) return false;
            if (getClass() != obj.getClass()) return false;
            final DelimitedStringImpl other = (DelimitedStringImpl) obj;
            return Objects.equals(this.identifier, other.identifier);
        }
        
        private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
            in.defaultReadObject();
            initParts();
        }
    }
    
    abstract class DelimitedStringImpl extends BaseImpl implements Serializable {
        public static final String DEFAULT_SEPARATOR = "/";

        public DelimitedStringImpl(String identifier) {
            super(identifier);
        }
        
        @Override
        protected final void initParts() {
            if (parts == null) {
                parts = identifier.split(getSeparator());
            }            
        }
        
        protected abstract String getSeparator();

    }
    
    abstract class URIImpl extends BaseImpl implements Serializable {
        private transient URI uri;

        public URIImpl(String identifier) {
            super(identifier);
            initUri();
        }

        public URIImpl(URI uri) {
            super(uri.toString());
            this.uri = uri;
        }

        public final URI getURI() {
            return uri;
        }
        
        private void initUri() {
            try {
                uri = new URI(identifier);
            } catch (URISyntaxException ex) {
                throw new IllegalArgumentException(ex);
            }            
        }
        
        private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
            in.defaultReadObject();
            initUri();
        }
    }
    
}
