/*
 * Decompiled with CFR 0.152.
 */
package nl.colorize.multimedialib.tool;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import nl.colorize.multimedialib.graphics.ImageAtlas;
import nl.colorize.multimedialib.graphics.ImageAtlasLoader;
import nl.colorize.multimedialib.math.MathUtils;
import nl.colorize.multimedialib.math.Rect;
import nl.colorize.multimedialib.renderer.java2d.Java2DImage;
import nl.colorize.multimedialib.tool.CommandLineTool;
import nl.colorize.util.FileUtils;
import nl.colorize.util.LogHelper;
import nl.colorize.util.swing.Utils2D;
import nl.colorize.util.xml.XMLHelper;
import org.jdom2.Document;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.Option;

public class ImageAtlasPacker
extends CommandLineTool {
    @Argument(index=0, metaVar="inputDir", required=true, usage="Location of images to pack")
    private String inputDirPath;
    @Argument(index=1, metaVar="outputDir", required=true, usage="Where to create files")
    private String outputDirPath;
    @Option(name="-atlasWidth", required=true, usage="Width of created image atlas")
    private int atlasWidth = 0;
    @Option(name="-outlines", required=false, usage="Draws sub-image outlines for testing purposes")
    private boolean outlines = false;
    private static final List<String> SUPPORTED_FORMATS = ImmutableList.of((Object)"jpg", (Object)"png");
    private static final int MAX_ATLAS_SIZE = 2048;
    private static final Color OUTLINE_COLOR = Color.PINK;
    private static final Logger LOGGER = LogHelper.getLogger(ImageAtlasPacker.class, (Handler[])new Handler[0]);

    public static void main(String[] args) {
        ImageAtlasPacker tool = new ImageAtlasPacker();
        tool.start(args);
    }

    @Override
    public void run() {
        File inputDir = this.parseInputDirectory(this.inputDirPath);
        List<File> imageFiles = this.findImageFilesToPack(inputDir);
        LOGGER.info("Found " + imageFiles.size() + " images to pack");
        for (File file : imageFiles) {
            LOGGER.info("    - " + FileUtils.getRelativePath((File)file, (File)inputDir));
        }
        LinkedHashMap<String, BufferedImage> subImages = new LinkedHashMap<String, BufferedImage>();
        for (File imageFile : imageFiles) {
            subImages.put(imageFile.getName(), this.loadImage(imageFile));
        }
        ImageAtlas imageAtlas = this.createImageAtlas(subImages, this.atlasWidth);
        int atlasHeight = imageAtlas.getSourceImage().getHeight();
        LOGGER.info("Image atlas dimensions: " + this.atlasWidth + "x" + atlasHeight);
        if (this.atlasWidth > 2048 || atlasHeight > 2048) {
            LOGGER.warning("Image atlas exceeds maximum size: " + this.atlasWidth + "x" + atlasHeight);
        }
        File outputDir = this.parseOutputDirectory(this.outputDirPath, true);
        this.saveImageAtlas(imageAtlas, outputDir);
    }

    private List<File> findImageFilesToPack(File inputDir) {
        try {
            return Files.walk(inputDir.toPath(), new FileVisitOption[0]).map(path -> path.toFile()).filter(file -> !file.isDirectory()).filter(file -> SUPPORTED_FORMATS.contains(FileUtils.getFileExtension((File)file).toLowerCase())).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Cannot read from " + inputDir.getAbsolutePath());
        }
    }

    private BufferedImage loadImage(File imageFile) {
        try {
            return Utils2D.loadImage((File)imageFile);
        }
        catch (IOException e) {
            throw new RuntimeException("Cannot load image: " + imageFile.getAbsolutePath());
        }
    }

    protected ImageAtlas createImageAtlas(Map<String, BufferedImage> subImages, int atlasWidth) {
        int cellSize = this.autoDetectCellSize(subImages.values());
        int atlasHeight = this.autoDetectAtlasHeight(subImages.values(), atlasWidth, cellSize);
        Map<BufferedImage, Rect> subImageBounds = this.calculateSubImageBounds(subImages.values(), atlasWidth, cellSize);
        BufferedImage atlas = this.renderImageAtlas(subImageBounds, atlasWidth, atlasHeight);
        ImageAtlas imageAtlas = new ImageAtlas(new Java2DImage(atlas));
        for (BufferedImage subImage : this.arrangeSubImages(subImages.values())) {
            Rect cell = subImageBounds.get(subImage);
            Rect region = new Rect(cell.getCenterX() - subImage.getWidth() / 2, cell.getCenterY() - subImage.getHeight() / 2, subImage.getWidth(), subImage.getHeight());
            imageAtlas.markSubImage(this.findSubImageName(subImages, subImage), region);
        }
        return imageAtlas;
    }

    private Iterable<BufferedImage> arrangeSubImages(Collection<BufferedImage> subImages) {
        ArrayList<BufferedImage> arrangedSubImages = new ArrayList<BufferedImage>(subImages);
        Collections.sort(arrangedSubImages, new Comparator<BufferedImage>(){

            @Override
            public int compare(BufferedImage a, BufferedImage b) {
                return a.getHeight() - b.getHeight();
            }
        });
        return arrangedSubImages;
    }

    private int autoDetectCellSize(Collection<BufferedImage> images) {
        int largestWidth = 0;
        int largestHeight = 0;
        for (BufferedImage image : images) {
            largestWidth = Math.max(largestWidth, MathUtils.nextPowerOfTwo(image.getWidth()));
            largestHeight = Math.max(largestHeight, MathUtils.nextPowerOfTwo(image.getHeight()));
        }
        return Math.max(largestWidth, largestHeight);
    }

    private int autoDetectAtlasHeight(Collection<BufferedImage> images, int atlasWidth, int cellSize) {
        int imagesPerRow = atlasWidth / cellSize;
        int rows = MathUtils.ceiling((float)images.size() / (float)imagesPerRow);
        return MathUtils.nextPowerOfTwo(rows * cellSize);
    }

    private Map<BufferedImage, Rect> calculateSubImageBounds(Collection<BufferedImage> images, int atlasWidth, int cellSize) {
        LinkedHashMap<BufferedImage, Rect> subImageBounds = new LinkedHashMap<BufferedImage, Rect>();
        int x = 0;
        int y = 0;
        for (BufferedImage image : images) {
            Preconditions.checkArgument((image.getWidth() <= cellSize && image.getHeight() <= cellSize ? 1 : 0) != 0, (Object)("Sub-image does not fit in cell: " + image.getWidth() + "x" + image.getHeight()));
            subImageBounds.put(image, new Rect(x, y, cellSize, cellSize));
            if ((x += cellSize) < atlasWidth) continue;
            x = 0;
            y += cellSize;
        }
        return subImageBounds;
    }

    private String findSubImageName(Map<String, BufferedImage> subImages, BufferedImage subImage) {
        for (Map.Entry<String, BufferedImage> entry : subImages.entrySet()) {
            if (entry.getValue() != subImage) continue;
            return entry.getKey();
        }
        throw new IllegalArgumentException("Unknown sub-image");
    }

    private BufferedImage renderImageAtlas(Map<BufferedImage, Rect> subImageBounds, int atlasWidth, int atlasHeight) {
        BufferedImage atlas = new BufferedImage(atlasWidth, atlasHeight, 2);
        Graphics2D g2 = Utils2D.createGraphics((BufferedImage)atlas, (boolean)true, (boolean)true);
        for (BufferedImage image : subImageBounds.keySet()) {
            Rect cell = subImageBounds.get(image);
            int cellCenterX = cell.getX() + cell.getWidth() / 2 - image.getWidth() / 2;
            int cellCenterY = cell.getY() + cell.getHeight() / 2 - image.getHeight() / 2;
            g2.drawImage((Image)image, cellCenterX, cellCenterY, null);
            if (!this.outlines) continue;
            g2.setColor(OUTLINE_COLOR);
            g2.drawRect(cellCenterX - cell.getWidth() / 2, cellCenterY - cell.getHeight() / 2, cell.getWidth(), cell.getHeight());
        }
        g2.dispose();
        return atlas;
    }

    protected void saveImageAtlas(ImageAtlas imageAtlas, File outputDir) {
        long timestamp = System.currentTimeMillis();
        File imageAtlasFile = new File(outputDir, "image_atlas_" + timestamp + ".png");
        File xmlFile = new File(outputDir, "image_atlas_" + timestamp + ".xml");
        try {
            Utils2D.savePNG((BufferedImage)((Java2DImage)imageAtlas.getSourceImage()).getImage(), (File)imageAtlasFile);
            LOGGER.info("Created image atlas at " + imageAtlasFile.getAbsolutePath());
            ImageAtlasLoader imageAtlasLoader = new ImageAtlasLoader(null);
            XMLHelper.write((Document)imageAtlasLoader.export(imageAtlas), (File)xmlFile);
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, "Could not create image atlas", e);
        }
    }
}

