//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

var canvas = null;
var context = null;

var imageContainer = null;
var images = {};
var audioContainer = null;
var audio = {};
var fontContainer = null;
var resourceContainer = null;

var pointerEventBuffer = [];
var keyStates = {};

var socket = null;

var maskCache = {};
var imageDataCache = {};

var CONSOLE_ENABLED = window.location.href.indexOf("console=true") != -1;

document.addEventListener("DOMContentLoaded", event => {
    imageContainer = document.getElementById("imageContainer");
    audioContainer = document.getElementById("audioContainer");
    fontContainer = document.getElementById("fontContainer");
    resourceContainer = document.getElementById("resourceContainer");

    var container = document.getElementById("multimediaLibContainer");
    initCanvas(container);
    main();
});

function initCanvas(container) {
    var width = Math.round(container.offsetWidth);
    var height = Math.round(document.documentElement.clientHeight);

    canvas = createCanvas(width, height);
    container.appendChild(canvas);
    context = createContext(canvas);

    canvas.addEventListener("mousedown", event => registerMouseEvent("mousedown", event), false);
    canvas.addEventListener("mouseup", event => registerMouseEvent("mouseup", event), false);
    canvas.addEventListener("mousemove", event => registerMouseEvent("mousemove", event), false);
    window.addEventListener("mouseout", event => registerMouseEvent("mouseout", event), false);
    canvas.addEventListener("touchstart", event => registerTouchEvent("touchstart", event), false);
    canvas.addEventListener("touchend", event => registerTouchEvent("touchend", event), false);
    canvas.addEventListener("touchmove", event => registerTouchEvent("touchmove", event), false);
    window.addEventListener("touchcancel", event => registerTouchEvent("touchcancel", event), false);
    window.addEventListener("keydown", onKeyDown);
    window.addEventListener("keyup", onKeyUp);
    window.addEventListener("resize", () => resizeCanvas(container));

    document.getElementById("loading").style.display = "none";
}

function createCanvas(width, height) {
    var newCanvas = document.createElement("canvas");
    newCanvas.style.width = width + "px";
    newCanvas.style.height = height + "px";
    newCanvas.width = width * window.devicePixelRatio;
    newCanvas.height = height * window.devicePixelRatio;
    return newCanvas;
}

function createContext(newCanvas) {
    return newCanvas.getContext("2d");
}

function resizeCanvas(container) {
    var targetWidth = Math.round(container.offsetWidth);
    var targetHeight = Math.round(document.documentElement.clientHeight);

    canvas.style.width = targetWidth + "px";
    canvas.style.height = targetHeight + "px";
    canvas.width = targetWidth * window.devicePixelRatio;
    canvas.height = targetHeight * window.devicePixelRatio;
}

function onFrame(callback) {
    context.clearRect(0, 0, canvas.width, canvas.height);
    callback();

    window.requestAnimationFrame(() => onFrame(callback));
}

function drawRect(x, y, width, height, color, alpha) {
    context.globalAlpha = alpha;
    context.fillStyle = color;
    context.fillRect(x, y, width, height);
    context.globalAlpha = 1.0;
}

function drawCircle(x, y, radius, color, alpha) {
    context.globalAlpha = alpha;
    context.fillStyle = color;
    context.beginPath();
    context.arc(x, y, radius, 0, 2.0 * Math.PI);
    context.fill();
    context.globalAlpha = 1.0;
}

function drawPolygon(points, color, alpha) {
    context.fillStyle = color;
    context.globalAlpha = alpha;
    context.beginPath();
    context.moveTo(points[0], points[1]);
    for (var i = 2; i < points.length; i += 2) {
        context.lineTo(points[i], points[i + 1]);
    }
    context.fill();
    context.globalAlpha = 1.0;
}

function drawImage(id, x, y, width, height, alpha, mask) {
    if (images[id]) {
        drawImageRegion(id, 0, 0, images[id].width, images[id].height, x, y, width, height,
            0.0, 1.0, 1.0, alpha, mask);
    }
}

function drawImageRegion(id, regionX, regionY, regionWidth, regionHeight, x, y, width, height,
                         rotation, scaleX, scaleY, alpha, mask) {
    if (images[id]) {
        var image = prepareImage(images[id], mask);

        context.save();
        context.globalAlpha = alpha;
        context.translate(x, y);
        context.rotate(rotation);
        context.scale(scaleX, scaleY);
        context.drawImage(image, regionX, regionY, regionWidth, regionHeight,
                          -width / 2.0, -height / 2.0, width, height);
        context.globalAlpha = 1.0;
        context.restore();
    }
}

function prepareImage(image, mask) {
    if (!mask) {
        return image;
    }

    var cacheKey = image.width + "x" + image.height;
    var maskImageCanvas = maskCache[cacheKey];

    if (maskImageCanvas == null) {
        maskImageCanvas = createCanvas(image.width, image.height);
        maskCache[cacheKey] = maskImageCanvas;
    }

    var maskImageContext = createContext(maskImageCanvas);
    maskImageContext.drawImage(image, 0, 0, image.width, image.height);
    maskImageContext.globalCompositeOperation = "source-atop";
    maskImageContext.fillStyle = mask;
    maskImageContext.fillRect(0, 0, image.width, image.height);

    return maskImageCanvas;
}

function drawText(text, font, size, color, bold, x, y, align, alpha) {
    context.globalAlpha = alpha;
    context.fillStyle = color;
    context.font = (bold ? "bold " : "") + size + "px " + font;
    context.textAlign = align;
    context.fillText(text, x, y);
    context.globalAlpha = 1.0;
}

function registerMouseEvent(eventType, event) {
    var mouseX = event.pageX - canvas.offsetLeft;
    var mouseY = event.pageY - canvas.offsetTop;
    registerPointerEvent(eventType + ";mouse;" + mouseX + ";" + mouseY);
    cancelEvent(event);
}

function registerTouchEvent(eventType, event) {
    for (var i = 0; i < event.changedTouches.length; i++) {
        var identifier = event.changedTouches[i].identifier;
        var touchX = event.changedTouches[i].pageX - canvas.offsetLeft;
        var touchY = event.changedTouches[i].pageY - canvas.offsetTop;
        registerPointerEvent(eventType + ";" + identifier + ";" + touchX + ";" + touchY);
    }
}

function registerPointerEvent(entry) {
    pointerEventBuffer.push(entry);
    if (CONSOLE_ENABLED && entry.indexOf("move") == -1) {
        logConsoleEvent(entry);
    }
}

function flushPointerEventBuffer() {
    var flushed = pointerEventBuffer;
    pointerEventBuffer = [];
    return flushed;
}

function onKeyDown(event) {
    keyStates[event.keyCode] = 1;
}

function onKeyUp(event) {
    keyStates[event.keyCode] = 0;
}

function cancelEvent(event) {
    event.preventDefault();
    event.stopPropagation();
}

function loadImage(id, path) {
    var imageElement = document.createElement("img");
    imageElement.src = path;
    imageContainer.appendChild(imageElement);
    images[id] = imageElement;
}

function loadAudio(id, path) {
    audio[id] = new Audio(path);
}

function loadFont(id, path, fontFamily) {
    var css = "";
    css += "@font-face { ";
    css += "    font-family: '" + fontFamily + "'; ";
    css += "    font-style: normal; ";
    css += "    font-weight: 400; ";
    css += "    src: url('" + path + "') format('truetype'); ";
    css += "}; ";

    var style = document.createElement("style");
    style.type = "text/css";
    style.appendChild(document.createTextNode(css));
    fontContainer.appendChild(style);
}

function getImageData(id, x, y) {
    if (images[id]) {
        var image = images[id];
        var imageData = imageDataCache[id];

        if (imageData == null) {
            var imageCanvas = createCanvas(image.width, image.height);
            var imageCanvasContext = imageCanvas.getContext("2d");
            imageCanvasContext.drawImage(image, 0, 0);
            imageData = imageCanvasContext;

            if (isImageDataAvailable(imageData, x, y)) {
                imageDataCache[id] = imageData;
            }
        }

        return imageData.getImageData(x, y, 1, 1).data;
    } else {
        return [-1, -1, -1, 255];
    }
}

function isImageDataAvailable(imageData, x, y) {
    var rgba = imageData.getImageData(x, y, 1, 1).data;
    return (rgba[0] + rgba[1] + rgba[2] + rgba[3]) != 0;
}

function playAudio(id, volume, loop) {
    audio[id].volume = volume;
    audio[id].loop = loop;
    audio[id].play();
}

function stopAudio(id, reset) {
    audio[id].pause();
    if (reset) {
        audio[id].currentTime = 0.0;
    }
}

function sendGetRequest(url, headers, callback) {
    var request = new XMLHttpRequest();
    request.onreadystatechange = () => {
        if (request.readyState == XMLHttpRequest.DONE) {
            callback(request.responseText);
        }
    };
    request.open("GET", url, true);
    prepareRequest(request, headers);
    request.send();
}

function sendPostRequest(url, headers, params, callback) {
    var request = new XMLHttpRequest();
    request.onreadystatechange = () => {
        if (request.readyState == XMLHttpRequest.DONE) {
            callback(request.responseText);
        }
    };
    request.open("POST", url, true);
    prepareRequest(request, headers);
    request.send(params);
}

function prepareRequest(request, headers) {
    request.setRequestHeader("X-Requested-With", "MultimediaLib");
    for (var i = 0; i < headers.length; i += 2) {
        request.setRequestHeader(headers[i], headers[i + 1]);
    }
}

function isWebSocketSupported() {
    return "WebSocket" in window;
}

function connectWebSocket(uri, callback) {
    socket = new WebSocket(uri);
    socket.onopen = event => callback("__open");
    socket.onmessage = event => callback(event.data);
    socket.onerror = error => console.log("Web socket error: " + error.message);
}

function sendWebSocket(message) {
    if (socket == null) {
        throw "Web socket not open";
    }

    socket.send(message);
}

function closeWebSocket() {
    if (socket != null) {
        socket.close();
        socket = null;
    }
}

function logConsoleEvent(message) {
    console.log(message);

    if (CONSOLE_ENABLED) {
        var browserConsole = document.getElementById("console");
        browserConsole.innerHTML = "<div>" + message + "</div>" + browserConsole.innerHTML;
    }
}

function takeScreenshot() {
    if (canvas == null) {
        throw "Canvas not yet initialized";
    }
    return canvas.toDataURL();
}
