//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.renderer.teavm;

import nl.colorize.multimedialib.renderer.InternetAccess;
import nl.colorize.multimedialib.renderer.WebSocketConnection;
import nl.colorize.util.Promise;
import nl.colorize.util.http.Headers;
import nl.colorize.util.http.PostData;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

/**
 * Sends HTTP requests by delegating them to JavaScript and sending them as
 * AJAX requests. This does mean that any requests sent from the application
 * must be allowed by the CORS (Cross Origin Resource Sharing) headers returned
 * by the server.
 * <p>
 * Web Sockets are also supported through the JavaScript web socket API, but
 * this is only supported when running in browsers that include this API.
 */
public class TeaInternetAccess implements InternetAccess {

    @Override
    public Promise<String> get(String url, Headers headers) {
        return new Promise<String>(promise -> {
            String[] headersArray = serializeHeaders(headers);
            Browser.sendGetRequest(url, headersArray, promise::resolve);
        });
    }

    @Override
    public Promise<String> post(String url, Headers headers, PostData data) {
        return new Promise<String>(promise -> {
            String[] headersArray = serializeHeaders(headers);
            String body = data.encode(StandardCharsets.UTF_8);
            Browser.sendPostRequest(url, headersArray, body, promise::resolve);
        });
    }

    private String[] serializeHeaders(Headers headers) {
        List<String> entries = new ArrayList<>();

        for (String name : headers.getNames()) {
            for (String value : headers.getValues(name)) {
                entries.add(name);
                entries.add(value);
            }
        }

        return entries.toArray(new String[0]);
    }

    @Override
    public boolean isWebSocketSupported() {
        return Browser.isWebSocketSupported();
    }

    @Override
    public WebSocketConnection connectWebSocket(String uri) {
        BrowserWebSocketConnection connection = new BrowserWebSocketConnection();
        Browser.connectWebSocket(uri, connection);
        return connection;
    }

    /**
     * Implementation of the WebSocketConnection interface that delegates all
     * operations to JavaScript.
     */
    private static class BrowserWebSocketConnection implements WebSocketConnection, WebSocketCallback {

        private boolean open = false;
        private List<String> messageBuffer = new ArrayList<>();
        private Consumer<String> callback;

        @Override
        public void onMessage(String message) {
            if (message.equals("__open")) {
                open = true;
                messageBuffer.forEach(this::send);
                messageBuffer.clear();
            } else if (callback != null) {
                callback.accept(message);
            }
        }

        @Override
        public void send(String message) {
            if (open) {
                Browser.sendWebSocket(message);
            } else {
                messageBuffer.add(message);
            }
        }

        @Override
        public void listen(Consumer<String> callback) {
            this.callback = callback;
        }

        @Override
        public void close() {
            open = false;
            Browser.closeWebSocket();
        }
    }
}
