//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.renderer.teavm;

import nl.colorize.multimedialib.renderer.ApplicationData;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.InputDevice;
import nl.colorize.multimedialib.renderer.InternetAccess;
import nl.colorize.multimedialib.renderer.MediaLoader;
import nl.colorize.multimedialib.renderer.Renderer;
import nl.colorize.multimedialib.renderer.Renderable;
import nl.colorize.multimedialib.renderer.Updatable;
import nl.colorize.util.Platform;

import java.util.ArrayList;
import java.util.List;

/**
 * Renderer based on TeaVM (http://teavm.org) that is transpiled to JavaScript
 * and runs in the browser.
 */
public class TeaRenderer implements Renderer, AnimationFrameCallback {

    private Canvas canvas;
    private TeaGraphicsContext graphics;
    private TeaInputDevice inputDevice;
    private TeaMediaLoader mediaLoader;
    private boolean isMediaRequired;
    private List<Updatable> updateCallbacks;
    private List<Renderable> renderCallbacks;

    private static final int HTML_CANVAS_ANIMATION_FRAMERATE = 60;

    public TeaRenderer(Canvas canvas) {
        this.canvas = canvas;
        this.graphics = new TeaGraphicsContext(canvas);
        this.inputDevice = new TeaInputDevice(canvas, getPlatform());
        this.mediaLoader = new TeaMediaLoader();
        this.isMediaRequired = false;
        this.updateCallbacks = new ArrayList<>();
        this.renderCallbacks = new ArrayList<>();

        Browser.renderFrame(this);
    }

    @Override
    public Canvas getCanvas() {
        return canvas;
    }

    @Override
    public InputDevice getInputDevice() {
        return inputDevice;
    }

    @Override
    public MediaLoader getMediaLoader() {
        return mediaLoader;
    }

    @Override
    public ApplicationData getApplicationData(String appName) {
        return new TeaLocalStorage();
    }

    @Override
    public InternetAccess getInternetAccess() {
        return new TeaInternetAccess();
    }

    @Override
    public void addUpdateCallback(Updatable callback) {
        updateCallbacks.add(callback);
    }

    @Override
    public void addRenderCallback(Renderable callback) {
        renderCallbacks.add(callback);
    }

    @Override
    public void onRenderFrame() {
        float deltaTime = 1f / HTML_CANVAS_ANIMATION_FRAMERATE;

        updateCanvas();
        inputDevice.update(deltaTime);

        if (isReady()) {
            for (Updatable updateCallback : updateCallbacks) {
                updateCallback.update(deltaTime);
            }

            for (Renderable renderCallback : renderCallbacks) {
                renderCallback.render(graphics);
            }
        }
    }

    private boolean isReady() {
        int canvasWidth = Math.round(Browser.getCanvasWidth());
        int canvasHeight = Math.round(Browser.getCanvasHeight());

        if (canvasWidth > 0 || canvasHeight > 0) {
            return !isMediaRequired || mediaLoader.isImageLoadingComplete();
        } else {
            return false;
        }
    }

    private void updateCanvas() {
        int canvasWidth = Math.round(Browser.getCanvasWidth());
        int canvasHeight = Math.round(Browser.getCanvasHeight());

        if (canvasWidth > 0 && canvasHeight > 0) {
            canvas.resizeScreen(canvasWidth, canvasHeight);
        }
    }

    /**
     * Requires all media to be fully loaded before the renderer starts the
     * animation loop. The browser loads media files asynchronously, so by
     * default the animation loop is already active at a time when not all
     * media files have been fully loaded yet.
     */
    public void requireMediaLoadingComplete() {
        isMediaRequired = true;
    }

    /**
     * Returns the display name of the current platform. This method is similar
     * to {@code Platform.getPlatformName()}, but detects the platform based on
     * the browser's {@code User-Agent} header rather than from the system
     * properties.
     */
    public Platform.PlatformFamily getPlatform() {
        String userAgent = Browser.getUserAgent().toLowerCase();

        if (userAgent.contains("iphone") || userAgent.contains("ipad")) {
            return Platform.PlatformFamily.IOS;
        } else if (userAgent.contains("android")) {
            return Platform.PlatformFamily.ANDROID;
        } else if (userAgent.contains("mac")) {
            return Platform.PlatformFamily.MAC;
        } else {
            return Platform.PlatformFamily.WINDOWS;
        }
    }

    @Override
    public String takeScreenshot() {
        return Browser.takeScreenshot();
    }
}
