//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.scene.ui;

import com.google.common.base.Preconditions;
import nl.colorize.multimedialib.graphics.Align;
import nl.colorize.multimedialib.graphics.Image;
import nl.colorize.multimedialib.graphics.TTFont;
import nl.colorize.multimedialib.math.Rect;
import nl.colorize.multimedialib.renderer.GraphicsContext;
import nl.colorize.multimedialib.renderer.InputDevice;

/**
 * Simple button widget with graphics and interaction entirely managed by the
 * renderer, not using the native widget. Buttons can consist of a flat
 * background or an image background, combined with a text label.
 */
public class Button extends Widget {

    private Image backgroundImage;
    private String label;
    private TTFont font;

    private InputDevice input;
    private Runnable onClick;

    public Button(Location location, Image background, String label, TTFont font) {
        super(location);
        this.backgroundImage = background;
        this.label = label;
        this.font = font;
    }

    public Button(Location location, Image background) {
        this(location, background, null, null);
    }

    public void setClickHandler(InputDevice input, Runnable onClick) {
        this.input = input;
        this.onClick = onClick;
    }

    @Override
    public void update(float deltaTime) {
        Preconditions.checkState(onClick != null, "Click handler has not been set");

        Rect bounds = Rect.around(getX(), getY(), backgroundImage.getWidth(), backgroundImage.getHeight());

        if (input.isPointerReleased() && bounds.contains(input.getPointer())) {
            onClick.run();
        }
    }

    @Override
    public void render(GraphicsContext graphics) {
        graphics.drawImage(backgroundImage, getX(), getY(), null);

        if (label != null && font != null) {
            graphics.drawText(label, font, getX(), getY() + backgroundImage.getHeight() * 0.2f,
                Align.CENTER);
        }
    }
}
