//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

let canvas = null;
let overlayCanvas = null;
let renderer = null;

let imageContainer = null;
let images = {};
let imageDataCache = {};
let audioContainer = null;
let audio = {};
let fontContainer = null;
let resourceContainer = null;

let pointerEventBuffer = [];
let keyStates = {};

let socket = null;

const WEBGL_ACTIVE = window.location.href.indexOf("webgl=true") != -1;
const WEBGL_SUPPORTED = window.WebGLRenderingContext != null && WEBGL_ACTIVE;
const CONSOLE_ENABLED = window.location.href.indexOf("console=true") != -1;

document.addEventListener("DOMContentLoaded", event => {
    imageContainer = document.getElementById("imageContainer");
    audioContainer = document.getElementById("audioContainer");
    fontContainer = document.getElementById("fontContainer");
    resourceContainer = document.getElementById("resourceContainer");

    let container = document.getElementById("multimediaLibContainer");
    initCanvas(container);
    main();
});

function initCanvas(container) {
    let width = Math.round(container.offsetWidth);
    let height = Math.round(document.documentElement.clientHeight);

    canvas = createCanvas(width, height);
    container.appendChild(canvas);

    renderer = createRenderer();
    initEventHandlers(container);

    if (renderer.hasOverlayCanvas()) {
        overlayCanvas = createCanvas(width, height);
        overlayCanvas.id = "overlayCanvas";
        container.appendChild(overlayCanvas);
    }

    document.getElementById("loading").style.display = "none";
}

function initEventHandlers(container) {
    canvas.addEventListener("mousedown", event => registerMouseEvent("mousedown", event), false);
    canvas.addEventListener("mouseup", event => registerMouseEvent("mouseup", event), false);
    canvas.addEventListener("mousemove", event => registerMouseEvent("mousemove", event), false);
    window.addEventListener("mouseout", event => registerMouseEvent("mouseout", event), false);
    canvas.addEventListener("touchstart", event => registerTouchEvent("touchstart", event), false);
    canvas.addEventListener("touchend", event => registerTouchEvent("touchend", event), false);
    canvas.addEventListener("touchmove", event => registerTouchEvent("touchmove", event), false);
    window.addEventListener("touchcancel", event => registerTouchEvent("touchcancel", event), false);
    window.addEventListener("keydown", onKeyDown);
    window.addEventListener("keyup", onKeyUp);
    window.addEventListener("resize", () => resizeCanvas(container));
}

function createCanvas(width, height) {
    let newCanvas = document.createElement("canvas");
    newCanvas.style.width = width + "px";
    newCanvas.style.height = height + "px";
    newCanvas.width = width * window.devicePixelRatio;
    newCanvas.height = height * window.devicePixelRatio;
    return newCanvas;
}

function createRenderer() {
    if (WEBGL_SUPPORTED) {
        let glContext = canvas.getContext("webgl");
        if (glContext != null) {
            console.log("Using WebGL renderer");
            return new WebGLRenderer(glContext);
        }
    }

    console.log("Using HTML5 canvas renderer");
    let context = canvas.getContext("2d");
    return new Html5CanvasRenderer(context);
}

function resizeCanvas(container) {
    let targetWidth = Math.round(container.offsetWidth);
    let targetHeight = Math.round(document.documentElement.clientHeight);

    canvas.style.width = targetWidth + "px";
    canvas.style.height = targetHeight + "px";
    canvas.width = targetWidth * window.devicePixelRatio;
    canvas.height = targetHeight * window.devicePixelRatio;

    if (overlayCanvas != null) {
        overlayCanvas.style.width = targetWidth + "px";
        overlayCanvas.style.height = targetHeight + "px";
        overlayCanvas.width = targetWidth * window.devicePixelRatio;
        overlayCanvas.height = targetHeight * window.devicePixelRatio;
    }
}

/**
 * Starts the animation loop. This method is called from TeaVM once the
 * application has been initialized. It will keep calling onFrame() to
 * perform frame updates.
 */
function startAnimationLoop(callback) {
    onFrame(callback, true);
}

/**
 * Callback method that is called every frame during the animation loop.
 * This will in turn call the callback method provided through TeaVM to
 * perform updates and rendering.
 */
function onFrame(callback, render) {
    let frameTime = 1.0 / 60;
    let limitedRendering = navigator.userAgent.indexOf("Firefox") != -1;
    let renderNextFrame = !limitedRendering || !render;

    if (render) {
        renderer.clearCanvas();
        callback(frameTime, true, true);
    } else {
        callback(frameTime, true, false);
    }

    window.requestAnimationFrame(() => onFrame(callback, renderNextFrame));
}

function registerMouseEvent(eventType, event) {
    let mouseX = event.pageX - canvas.offsetLeft;
    let mouseY = event.pageY - canvas.offsetTop;
    registerPointerEvent(eventType + ";mouse;" + mouseX + ";" + mouseY);
    cancelEvent(event);
}

function registerTouchEvent(eventType, event) {
    for (let i = 0; i < event.changedTouches.length; i++) {
        let identifier = event.changedTouches[i].identifier;
        let touchX = event.changedTouches[i].pageX - canvas.offsetLeft;
        let touchY = event.changedTouches[i].pageY - canvas.offsetTop;
        registerPointerEvent(eventType + ";" + identifier + ";" + touchX + ";" + touchY);
    }
}

function registerPointerEvent(entry) {
    pointerEventBuffer.push(entry);
    if (CONSOLE_ENABLED && entry.indexOf("move") == -1) {
        logConsoleEvent(entry);
    }
}

function flushPointerEventBuffer() {
    let flushed = pointerEventBuffer;
    pointerEventBuffer = [];
    return flushed;
}

function onKeyDown(event) {
    keyStates[event.keyCode] = 1;
    cancelEvent(event);
}

function onKeyUp(event) {
    keyStates[event.keyCode] = 0;
    cancelEvent(event);
}

function cancelEvent(event) {
    event.preventDefault();
    event.stopPropagation();
}

function loadImage(id, path) {
    let imageElement = document.createElement("img");
    imageElement.onload = () => renderer.onLoadImage(id, imageElement);
    imageElement.src = path;
    imageContainer.appendChild(imageElement);
    images[id] = imageElement;
}

function loadAudio(id, path) {
    audio[id] = new Audio(path);
}

function loadFont(id, path, fontFamily) {
    let css = "";
    css += "@font-face { ";
    css += "    font-family: '" + fontFamily + "'; ";
    css += "    font-style: normal; ";
    css += "    font-weight: 400; ";
    css += "    src: url('" + path + "') format('truetype'); ";
    css += "}; ";

    let style = document.createElement("style");
    style.type = "text/css";
    style.appendChild(document.createTextNode(css));
    fontContainer.appendChild(style);
}

function getImageData(id, x, y) {
    if (images[id]) {
        let image = images[id];
        let imageData = imageDataCache[id];

        if (imageData == null) {
            let imageCanvas = createCanvas(image.width, image.height);
            let imageCanvasContext = imageCanvas.getContext("2d");
            imageCanvasContext.drawImage(image, 0, 0);
            imageData = imageCanvasContext;

            if (isImageDataAvailable(imageData, x, y)) {
                imageDataCache[id] = imageData;
            }
        }

        return imageData.getImageData(x, y, 1, 1).data;
    } else {
        return [-1, -1, -1, 255];
    }
}

function isImageDataAvailable(imageData, x, y) {
    let rgba = imageData.getImageData(x, y, 1, 1).data;
    return (rgba[0] + rgba[1] + rgba[2] + rgba[3]) != 0;
}

/**
 * Converts a hexadecimal color to an array of red, green, and blue components
 * represented by an integer between 0 and 255. For example, #FF0000 will return
 * the array [255, 0, 0].
 */
function toRGB(hexColor) {
    if (hexColor.length != 7) {
        throw "Invalid hexadecimal color: " + hexColor;
    }

    return [
        parseInt(hexColor.substring(1, 3), 16),
        parseInt(hexColor.substring(3, 5), 16),
        parseInt(hexColor.substring(5, 7), 16)
    ];
}

/**
 * Converts a hexadecimal color and alpha channel into an array with 4 elements,
 * each representd by a value between 0.0 and 1.0.
 */
function toRGBA(hexColor, alpha) {
    let rgb = toRGB(hexColor);
    return [rgb[0] / 255.0, rgb[1] / 255.0, rgb[2] / 255.0, alpha];
}

function playAudio(id, volume, loop) {
    audio[id].volume = volume;
    audio[id].loop = loop;
    audio[id].play();
}

function stopAudio(id, reset) {
    audio[id].pause();
    if (reset) {
        audio[id].currentTime = 0.0;
    }
}

function sendGetRequest(url, headers, callback) {
    let request = new XMLHttpRequest();
    request.onreadystatechange = () => {
        if (request.readyState == XMLHttpRequest.DONE) {
            callback(request.responseText);
        }
    };
    request.open("GET", url, true);
    prepareRequest(request, headers);
    request.send();
}

function sendPostRequest(url, headers, params, callback) {
    let request = new XMLHttpRequest();
    request.onreadystatechange = () => {
        if (request.readyState == XMLHttpRequest.DONE) {
            callback(request.responseText);
        }
    };
    request.open("POST", url, true);
    prepareRequest(request, headers);
    request.send(params);
}

function prepareRequest(request, headers) {
    request.setRequestHeader("X-Requested-With", "MultimediaLib");
    for (let i = 0; i < headers.length; i += 2) {
        request.setRequestHeader(headers[i], headers[i + 1]);
    }
}

function connectWebSocket(uri, callback) {
    socket = new WebSocket(uri);
    socket.onopen = event => callback("__open");
    socket.onmessage = event => callback(event.data);
    socket.onerror = error => console.log("Web socket error: " + error.message);
}

function sendWebSocket(message) {
    if (socket == null) {
        throw "Web socket not open";
    }

    socket.send(message);
}

function closeWebSocket() {
    if (socket != null) {
        socket.close();
        socket = null;
    }
}

function logConsoleEvent(message) {
    console.log(message);

    if (CONSOLE_ENABLED) {
        let browserConsole = document.getElementById("console");
        browserConsole.innerHTML = "<div>" + message + "</div>" + browserConsole.innerHTML;
    }
}

function takeScreenshot() {
    if (canvas == null) {
        throw "Canvas not yet initialized";
    }
    return canvas.toDataURL();
}
