//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

/**
 * Uses WebGL to draw graphics on the canvas. WebGL is supported by all modern
 * browsers, but may not be supported by the platform itself.
 */
class WebGLRenderer {

    constructor(glContext) {
        if (glContext == null) {
            throw "WebGL not supported";
        }

        this.gl = glContext;
        this.textures = {};

        this.gl.enable(this.gl.BLEND);
        this.gl.blendFunc(this.gl.ONE, this.gl.ONE_MINUS_SRC_ALPHA);

        this.shaderProgram = this.initShaderProgram();

        let buffer = this.gl.createBuffer();
        this.gl.bindBuffer(this.gl.ARRAY_BUFFER, buffer);
    }

    initShaderProgram() {
        let vertexShader = this.loadShader(this.gl.VERTEX_SHADER, this.initVertexShader());
        let fragmentShader = this.loadShader(this.gl.FRAGMENT_SHADER, this.initFragmentShader());

        let shaderProgram = this.gl.createProgram();
        this.gl.attachShader(shaderProgram, vertexShader);
        this.gl.attachShader(shaderProgram, fragmentShader);
        this.gl.linkProgram(shaderProgram);

        if (!this.gl.getProgramParameter(shaderProgram, this.gl.LINK_STATUS)) {
            throw "Error while initializing WebGL shader: " + this.gl.getProgramInfoLog(shaderProgram);
        }

        return shaderProgram;
    }

    loadShader(type, glsl) {
        let shader = this.gl.createShader(type);
        this.gl.shaderSource(shader, glsl);
        this.gl.compileShader(shader);

        if (!this.gl.getShaderParameter(shader, this.gl.COMPILE_STATUS)) {
            throw "Error while loading WebGL shader: " + this.gl.getShaderInfoLog(shader);
        }

        return shader;
    }

    initVertexShader() {
        return `
            attribute vec4 aPosition;
            attribute vec4 aColor;
            attribute vec2 aTextureCoordinates;
            
            varying vec4 vColor;
            varying vec2 vTextureCoordinates;
            
            uniform vec2 uRotation;
            uniform vec2 uScale;

            void main() {
                vColor = aColor;
                vTextureCoordinates = aTextureCoordinates;
                
                vec2 rotatedPosition = vec2(
                    aPosition.x * uRotation.y + aPosition.y * uRotation.x,
                    aPosition.y * uRotation.y - aPosition.x * uRotation.x
                );

                gl_Position = vec4(rotatedPosition * uScale, 0.0, 1.0);
            }
        `;
    }

    initFragmentShader() {
        return `
            precision mediump float;

            uniform sampler2D uTextureUnit;
            uniform vec4 uColor;
            uniform float uAlpha;
            
            varying vec2 vTextureCoordinates;
            
            void main() {
                if (vTextureCoordinates.x >= 0.0 && vTextureCoordinates.y >= 0.0) {
                    vec4 color = texture2D(uTextureUnit, vTextureCoordinates);
                    gl_FragColor = vec4(color.rgb, uAlpha * color.a);
                } else {
                    gl_FragColor = uColor;
                }
            }
        `;
    }

    hasOverlayCanvas() {
        return true;
    }

    onLoadImage(id, imageElement) {
        let texture = this.loadTexture(id);
        this.gl.bindTexture(this.gl.TEXTURE_2D, texture);
        this.gl.texImage2D(this.gl.TEXTURE_2D, 0, this.gl.RGBA, this.gl.RGBA,
            this.gl.UNSIGNED_BYTE, imageElement);
    }

    clearCanvas() {
        this.gl.viewport(0, 0, this.gl.canvas.width, this.gl.canvas.height);
        this.gl.clearColor(0.0, 0.0, 0.0, 1.0);
        this.gl.clear(this.gl.COLOR_BUFFER_BIT | this.gl.DEPTH_BUFFER_BIT);

        this.initOverlayContext();
        this.overlayContext.clearRect(0, 0, overlayCanvas.width, overlayCanvas.height);

        this.gl.useProgram(this.shaderProgram);
    }

    drawRect(x, y, width, height, color, alpha) {
        let textureCoordinates = {
            s0: -1,
            s1: -1,
            t0: -1,
            t1: -1
        };

        let vertexData = this.toVertexData(x, y, width, height, textureCoordinates);
        this.renderPolygon(vertexData, null, toRGBA(color, alpha), [0.0, 0.0], [1.0, 1.0]);
    }

    drawCircle(x, y, radius, color, alpha) {
        //TODO
        console.log("drawCircle not yet implemented in WebGL renderer");
    }

    drawPolygon(points, color, alpha) {
        //TODO
        console.log("drawPolygon not yet implemented in WebGL renderer");
    }

    drawImage(id, x, y, width, height, alpha, mask) {
        if (images[id]) {
            this.drawImageRegion(id, 0, 0, images[id].width, images[id].height,
                x, y, width, height, 0, 100, 100, alpha, mask);
        }
    }

    drawImageRegion(id, regionX, regionY, regionWidth, regionHeight, x, y, width, height,
                    rotation, scaleX, scaleY, alpha, mask) {
        if (images[id]) {
            let textureCoordinates = {
                s0: regionX / images[id].width,
                s1: (regionX + regionWidth) / images[id].width,
                t0: 1.0 - ((regionY + regionHeight) / images[id].height),
                t1: 1.0 - (regionY / images[id].height),
            };

            let vertexData = this.toVertexData(x - width / 2, y - height / 2, width, height,
                textureCoordinates);
            let texture = this.loadTexture(id);
            let color = toRGBA("#FFFFFF", alpha);
            let rotationVector = [Math.sin(rotation), Math.cos(rotation)];
            let scaleVector = [1.0, 1.0];

            this.renderPolygon(vertexData, texture, color, rotationVector, scaleVector);
        }
    }

    drawText(text, font, size, color, bold, x, y, align, alpha) {
        this.initOverlayContext();

        this.overlayContext.globalAlpha = alpha;
        this.overlayContext.fillStyle = color;
        this.overlayContext.font = (bold ? "bold " : "") + size + "px " + font;
        this.overlayContext.textAlign = align;
        this.overlayContext.fillText(text, x, y);
        this.overlayContext.globalAlpha = 1.0;
    }

    initOverlayContext() {
        if (this.overlayContext == null) {
            this.overlayContext = overlayCanvas.getContext("2d");
        }
    }

    toVertexData(x, y, width, height, textureCoordinates) {
        let centerS = textureCoordinates.s0 + (textureCoordinates.s1 - textureCoordinates.s0) / 2.0;
        let centerT = textureCoordinates.t1 + (textureCoordinates.t0 - textureCoordinates.t1) / 2.0;

        let vertexData = [
            x + width / 2, y + height / 2, centerS, centerT,
            x, y + height, textureCoordinates.s0, textureCoordinates.t1,
            x + width, y + height, textureCoordinates.s1, textureCoordinates.t1,
            x + width, y, textureCoordinates.s1, textureCoordinates.t0,
            x, y, textureCoordinates.s0, textureCoordinates.t0,
            x, y + height, textureCoordinates.s0, textureCoordinates.t1
        ];

        // Converts X and Y coordinates in an array of vertices from the canvas
        // coordinate space (0, 0, width, height) to the OpenGL ES coordinate
        // space (-1, 1, 1, -1).
        for (let i = 0; i < vertexData.length; i += 4) {
            vertexData[i] = (vertexData[i] / canvas.width * 2.0) - 1.0;
            vertexData[i + 1] = ((vertexData[i + 1] / canvas.height * 2.0) - 1.0) * -1.0;
        }

        return vertexData;
    }

    loadTexture(id) {
        if (this.textures[id]) {
            return this.textures[id];
        }

        let texture = this.gl.createTexture();
        this.gl.bindTexture(this.gl.TEXTURE_2D, texture);
        this.gl.texImage2D(this.gl.TEXTURE_2D, 0, this.gl.RGBA, 1, 1, 0, this.gl.RGBA,
            this.gl.UNSIGNED_BYTE, new Uint8Array([255, 255, 255, 255]));
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_WRAP_S, this.gl.CLAMP_TO_EDGE);
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_WRAP_T, this.gl.CLAMP_TO_EDGE);
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_MIN_FILTER, this.gl.LINEAR);
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_MAG_FILTER, this.gl.LINEAR);
        this.textures[id] = texture;
        return texture;
    }

    renderPolygon(vertexData, texture, colorVector, rotationVector, scaleVector) {
        let positionLocation = this.gl.getAttribLocation(this.shaderProgram, "aPosition");
        let rotationLocation = this.gl.getUniformLocation(this.shaderProgram, "uRotation");
        let scaleLocation = this.gl.getUniformLocation(this.shaderProgram, "uScale");
        let colorLocation = this.gl.getUniformLocation(this.shaderProgram, "uColor");
        let alphaLocation = this.gl.getUniformLocation(this.shaderProgram, "uAlpha");
        let textureCoordsLocation = this.gl.getAttribLocation(this.shaderProgram, "aTextureCoordinates");

        this.gl.uniform2fv(rotationLocation, rotationVector);
        this.gl.uniform2fv(scaleLocation, scaleVector);
        this.gl.uniform4fv(colorLocation, colorVector);
        this.gl.uniform1f(alphaLocation, colorVector[3]);
        this.gl.bufferData(this.gl.ARRAY_BUFFER, new Float32Array(vertexData), this.gl.STATIC_DRAW);

        if (texture != null) {
            this.gl.bindTexture(this.gl.TEXTURE_2D, texture);
        } else {
            this.gl.bindTexture(this.gl.TEXTURE_2D, null);
        }

        this.gl.vertexAttribPointer(positionLocation, 2, this.gl.FLOAT, false, 16, 0);
        this.gl.enableVertexAttribArray(positionLocation);
        this.gl.vertexAttribPointer(textureCoordsLocation, 2, this.gl.FLOAT, false, 16, 8);
        this.gl.enableVertexAttribArray(textureCoordsLocation);

        this.gl.drawArrays(this.gl.TRIANGLE_FAN, 0, 6);

        this.gl.disableVertexAttribArray(positionLocation);
        this.gl.disableVertexAttribArray(textureCoordsLocation);
    }
}
