//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.graphics;

import com.google.common.base.Splitter;
import nl.colorize.multimedialib.math.Point;
import nl.colorize.multimedialib.math.Rect;
import nl.colorize.multimedialib.renderer.GraphicsContext;
import nl.colorize.multimedialib.renderer.MediaException;
import nl.colorize.util.LogHelper;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

/**
 * Describes a font that consists of a number of images. This type of font is
 * generally used for environments that do not support rendering TrueType fonts,
 * though the performance of bitmap fonts also tends to be better.
 */
public class BitmapFont implements TextFont {

    private Image image;
    private String family;
    private int size;
    private int lineHeight;
    private int base;
    private Map<Character, CharLayout> chars;

    private static final Splitter LINE_SPLITTER = Splitter.on("\n");
    private static final Splitter TAG_SPLITTER = Splitter.on(" ").trimResults().omitEmptyStrings();
    private static final Logger LOGGER = LogHelper.getLogger(BitmapFont.class);

    private BitmapFont(Image image) {
        this.image = image;
        this.chars = new HashMap<>();
    }

    @Override
    public String getFamily() {
        return family;
    }

    @Override
    public int getSize() {
        return size;
    }

    protected int getLineHeight() {
        return lineHeight;
    }

    protected int getBase() {
        return base;
    }

    protected Rect getBounds(char c) {
        CharLayout layout = chars.get(c);
        return layout.bounds;
    }

    @Override
    public BitmapFont derive(int newSize) {
        LOGGER.warning("Deriving fonts not supported for bitmap fonts");
        return this;
    }

    @Override
    public BitmapFont derive(ColorRGB newColor) {
        LOGGER.warning("Deriving fonts not supported for bitmap fonts");
        return this;
    }

    @Override
    public BitmapFont deriveBold() {
        LOGGER.warning("Deriving fonts not supported for bitmap fonts");
        return this;
    }

    /**
     * Draws a text with this bitmap font using the specified graphics context.
     * This method will trigger several drawing operations to draw the various
     * sub-images.
     */
    public void draw(GraphicsContext context, String text, float x, float y, Align align,
                     AlphaTransform alpha) {
        Transform transform = alpha != null ? alpha.toTransform() : null;
        int lineWidth = measureLineWidth(text);
        Point cursor = resetCursor(x, y, lineWidth, align);

        for (int i = 0; i < text.length(); i++) {
            CharLayout layout = chars.get(text.charAt(i));
            if (layout != null) {
                context.drawImage(layout.subImage, cursor.getX() + layout.xOffset,
                    cursor.getY() + layout.yOffset, transform);
                cursor.setX(cursor.getX() + layout.xAdvance);
            }
        }
    }

    private Point resetCursor(float x, float y, int lineWidth, Align align) {
        switch (align) {
            case LEFT : return new Point(x, y);
            case RIGHT : return new Point(x - lineWidth, y);
            case CENTER : return new Point(x - lineWidth / 2f, y);
            default : throw new IllegalArgumentException("Unknown text alignment: " + align);
        }
    }

    private int measureLineWidth(String text) {
        int lineWidth = 0;

        for (int i = 0; i < text.length(); i++) {
            CharLayout layout = chars.get(text.charAt(i));
            if (layout != null) {
                lineWidth += layout.xAdvance;
            }
        }

        return lineWidth;
    }

    /**
     * Creates a bitmap font from an image and metadata parsed from a {@code .fnt}
     * file using the AngelCode font format. AngelCode fonts can be created using
     * various tools, such as libGDX Hiero (https://github.com/libgdx/libgdx/wiki/Hiero).
     *
     * @throws MediaException if the font metadata cannot be parsed.
     */
    public static BitmapFont parseFNT(Image image, String fntMetadata) {
        BitmapFont font = new BitmapFont(image);

        for (String line : LINE_SPLITTER.splitToList(fntMetadata)) {
            Map<String, String> tags = parseTags(line);

            if (line.startsWith("info ")) {
                font.family = tags.get("face");
                font.size = Integer.parseInt(tags.get("size"));
            } else if (line.startsWith("common ")) {
                font.lineHeight = Integer.parseInt(tags.get("lineHeight"));
                font.base = Integer.parseInt(tags.get("base"));
            } else if (line.startsWith("char ")) {
                CharLayout layout = new CharLayout();
                layout.bounds = new Rect(Integer.parseInt(tags.get("x")), Integer.parseInt(tags.get("y")),
                    Integer.parseInt(tags.get("width")), Integer.parseInt(tags.get("height")));
                layout.subImage = image.getRegion(layout.bounds);
                layout.xOffset = Integer.parseInt(tags.get("xoffset"));
                layout.yOffset = Integer.parseInt(tags.get("yoffset"));
                layout.xAdvance = Integer.parseInt(tags.get("xadvance"));

                font.chars.put((char) Integer.parseInt(tags.get("id")), layout);
            }
        }

        return font;
    }

    private static Map<String, String> parseTags(String line) {
        Map<String, String> tags = new HashMap<>();

        for (String tag : TAG_SPLITTER.splitToList(line)) {
            if (tag.indexOf('=') != -1) {
                String name = tag.substring(0, tag.indexOf('='));
                String value = tag.substring(tag.indexOf('=') + 1).replace("\"", "");
                tags.put(name, value);
            }
        }

        return tags;
    }

    /**
     * Display layout information related to one o the characters that can be
     * displayed using this font.
     */
    private static class CharLayout {

        private Image subImage;
        private Rect bounds;
        private int xOffset;
        private int yOffset;
        private int xAdvance;
    }
}
