//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.renderer.libgdx;

import com.badlogic.gdx.Gdx;
import com.google.common.collect.ImmutableList;
import nl.colorize.multimedialib.math.Point;
import nl.colorize.multimedialib.math.Rect;
import nl.colorize.multimedialib.renderer.InputDevice;
import nl.colorize.multimedialib.renderer.KeyCode;
import nl.colorize.multimedialib.renderer.Updatable;

import java.util.List;

/**
 * Access to libGDX's user input state.
 */
public class GDXInput implements InputDevice, Updatable {

    private Point pointer;
    private boolean pointerPressed;
    private boolean pointerReleased;

    protected GDXInput() {
        pointer = new Point(0f, 0f);
        pointerPressed = false;
        pointerReleased = false;
    }

    @Override
    public void update(float deltaTime) {
        pointer.set(Gdx.input.getX(), Gdx.input.getY());

        if (Gdx.input.isTouched()) {
            pointerPressed = true;
            pointerReleased = false;
        } else if (pointerPressed) {
            pointerPressed = false;
            pointerReleased = true;
        } else {
            pointerPressed = false;
            pointerReleased = false;
        }
    }

    @Override
    public List<Point> getPointers() {
        return ImmutableList.of(pointer);
    }

    @Override
    public boolean isPointerPressed(Rect area) {
        return pointerPressed && area.contains(pointer);
    }

    @Override
    public boolean isPointerReleased(Rect area) {
        return pointerReleased && area.contains(pointer);
    }

    @Override
    public boolean isTouchAvailable() {
        return false;
    }

    @Override
    public boolean isKeyboardAvailable() {
        return false;
    }

    @Override
    public boolean isKeyPressed(KeyCode keyCode) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean isKeyReleased(KeyCode keyCode) {
        throw new UnsupportedOperationException();
    }

    @Override
    public String requestTextInput(String label, String initialValue) {
        throw new UnsupportedOperationException();
    }
}
