//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.scene;

import com.google.common.base.Preconditions;
import nl.colorize.multimedialib.graphics.Image;
import nl.colorize.multimedialib.math.RotatingBuffer;
import nl.colorize.multimedialib.renderer.ApplicationData;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.FilePointer;
import nl.colorize.multimedialib.renderer.GraphicsContext;
import nl.colorize.multimedialib.renderer.InputDevice;
import nl.colorize.multimedialib.renderer.InternetAccess;
import nl.colorize.multimedialib.renderer.MediaLoader;
import nl.colorize.multimedialib.renderer.Renderable;
import nl.colorize.multimedialib.renderer.Renderer;
import nl.colorize.multimedialib.renderer.Updatable;
import nl.colorize.util.Stopwatch;
import nl.colorize.util.animation.Interpolation;
import nl.colorize.util.animation.Timeline;

/**
 * Implements a mechanism on top of the renderer that divides the application
 * life cycle into a number of separate scenes. One scene is marked as currently
 * active, and will receive frame updates and render graphics for as a long as
 * it is active. Scenes will also receive notifications whenever the active scene
 * changes.
 */
public class Application implements Updatable, Renderable {

    private Renderer renderer;
    private MediaCache mediaCache;

    private Scene activeScene;
    private Scene requestedScene;

    private boolean orientationLock;
    private Effect orientationLockAnim;

    private Stopwatch fpsTimer;
    private RotatingBuffer fpsBuffer;
    private RotatingBuffer frameTimeBuffer;

    private static final int FPS_MEASUREMENT_BUFFER_SIZE = 100;
    private static final FilePointer ORIENTATION_LOCK_IMAGE = new FilePointer("orientation-lock.png");

    public Application(Renderer renderer) {
        this.renderer = renderer;
        this.mediaCache = new MediaCache(renderer.getMediaLoader());

        this.orientationLock = false;
        this.orientationLockAnim = initOrientationLockAnim();

        this.fpsTimer = new Stopwatch();
        this.fpsBuffer = new RotatingBuffer(FPS_MEASUREMENT_BUFFER_SIZE);
        this.frameTimeBuffer = new RotatingBuffer(FPS_MEASUREMENT_BUFFER_SIZE);

        renderer.addUpdateCallback(this);
        renderer.addRenderCallback(this);
    }

    private Effect initOrientationLockAnim() {
        Image orientationLockImage = mediaCache.loadImage(ORIENTATION_LOCK_IMAGE);

        Timeline timeline = new Timeline(Interpolation.EASE, true);
        timeline.addKeyFrame(0f, 100f);
        timeline.addKeyFrame(1f, 110f);
        timeline.addKeyFrame(2f, 100f);

        Effect effect = Effect.forImage(orientationLockImage, timeline);
        effect.modify(value -> effect.getTransform().setScale(Math.round(value)));
        return effect;
    }

    /**
     * Requests to change the active scene after the current frame has been
     * completed.
     *
     * @throws IllegalStateException if a different scene has already been
     *         requested, but that scene has not yet started.
     */
    public void changeScene(Scene requestedScene) {
        Preconditions.checkState(this.requestedScene == null,
            "Another scene has already been requested: " + requestedScene);

        this.requestedScene = requestedScene;
    }

    public Scene getActiveScene() {
        return activeScene;
    }

    /**
     * Restricts the application to landscape orientation. When attempting to
     * use the application in portrait application, the current scene will be
     * suspended and an image will be shown prompting to switch back to
     * landscape orientation.
     */
    public void lockScreenOrientation() {
        orientationLock = true;
    }

    @Override
    public void update(float deltaTime) {
        if (requestedScene != null) {
            if (activeScene != null) {
                activeScene.end();
            }

            activeScene = requestedScene;
            requestedScene = null;
            activeScene.start();
        }

        Stopwatch frameTimer = new Stopwatch();
        frameTimer.tick();

        if (activeScene != null && isScreenOrientationSupported()) {
            activeScene.update(deltaTime);
        } else {
            orientationLockAnim.update(deltaTime);
        }

        long actualFrameTime = frameTimer.tick();
        frameTimeBuffer.add(actualFrameTime);
    }

    private boolean isScreenOrientationSupported() {
        if (!orientationLock) {
            return true;
        }

        Canvas canvas = renderer.getCanvas();
        return canvas.getWidth() > 0 && canvas.getWidth() > canvas.getHeight();
    }

    @Override
    public void render(GraphicsContext graphics) {
        if (activeScene != null) {
            long fpsValue = fpsTimer.tick();
            fpsBuffer.add(fpsValue);

            if (isScreenOrientationSupported()) {
                activeScene.render(graphics);
            } else {
                drawOrientationLock(graphics);
            }
        }
    }

    private void drawOrientationLock(GraphicsContext graphics) {
        Canvas canvas = getCanvas();
        orientationLockAnim.setPosition(canvas.getWidth() / 2f, canvas.getHeight() / 2f);
        orientationLockAnim.render(graphics);
    }

    /**
     * Returns the renderer that is used by this application.
     */
    public Renderer getRenderer() {
        return renderer;
    }

    public Canvas getCanvas() {
        return renderer.getCanvas();
    }

    public InputDevice getInputDevice() {
        return renderer.getInputDevice();
    }

    public MediaLoader getMediaLoader() {
        return mediaCache;
    }

    public ApplicationData getApplicationData(String appName) {
        return renderer.getApplicationData(appName);
    }

    public InternetAccess getInternetAccess() {
        return renderer.getInternetAccess();
    }

    public float getAverageFPS() {
        return 1000f / fpsBuffer.getAverageValue();
    }

    public float getAverageFrameTime() {
        return frameTimeBuffer.getAverageValue();
    }
}
