//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.scene;

import com.google.common.collect.ImmutableList;
import nl.colorize.multimedialib.graphics.ColorRGB;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.GraphicsContext;

import java.util.Collections;
import java.util.List;

/**
 * Scene that is split into a number of subsystems, which are also represented
 * by scenes. This is typically needed for scenes that contain a significant
 * amount of functionality, in which case the scene can be split into subsystems,
 * with each subsystem corresponding to a functional area, to manage the scene's
 * complexity.
 */
public abstract class ComplexScene implements Scene {

    private Application app;
    private List<Scene> subSystems;
    private EffectManager effectManager;
    private ColorRGB backgroundColor;

    public ComplexScene(Application app) {
        this.subSystems = Collections.emptyList();
        this.effectManager = new EffectManager();
    }

    @Override
    public final void start() {
        initScene();

        subSystems = new ImmutableList.Builder<Scene>()
            .addAll(initSubSystems())
            .add(effectManager)
            .build();

        for (Scene subSystem : subSystems) {
            subSystem.start();
        }
    }

    /**
     * Initializes the scene itself, before any of its subsystems are started.
     * This method is called every time the scene is started.
     */
    protected abstract void initScene();

    /**
     * Initializes the list of subsystems that will be used within this scene.
     * This method is called every time the scene is started.
     */
    protected abstract List<Scene> initSubSystems();

    @Override
    public void update(float deltaTime) {
        for (Scene subSystem : subSystems) {
            subSystem.update(deltaTime);
        }
    }

    @Override
    public void render(GraphicsContext graphics) {
        if (backgroundColor != null) {
            graphics.drawBackground(backgroundColor);
        }

        for (Scene subSystem : subSystems) {
            if (shouldRender(subSystem)) {
                subSystem.render(graphics);
            }
        }
    }

    protected boolean shouldRender(Scene subSystem) {
        return true;
    }

    protected Application getApp() {
        return app;
    }

    protected Canvas getCanvas() {
        return app.getCanvas();
    }

    public EffectManager getEffectManager() {
        return effectManager;
    }

    public void setBackgroundColor(ColorRGB backgroundColor) {
        this.backgroundColor = backgroundColor;
    }

    public ColorRGB getBackgroundColor() {
        return backgroundColor;
    }
}
