//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.tool;

import com.google.common.collect.ImmutableList;
import com.google.common.net.HttpHeaders;
import nl.colorize.multimedialib.graphics.Align;
import nl.colorize.multimedialib.graphics.Animation;
import nl.colorize.multimedialib.graphics.ColorRGB;
import nl.colorize.multimedialib.graphics.Image;
import nl.colorize.multimedialib.graphics.Sprite;
import nl.colorize.multimedialib.graphics.SpriteSheet;
import nl.colorize.multimedialib.graphics.TextFont;
import nl.colorize.multimedialib.graphics.Transform;
import nl.colorize.multimedialib.math.Point;
import nl.colorize.multimedialib.math.RandomGenerator;
import nl.colorize.multimedialib.math.Rect;
import nl.colorize.multimedialib.renderer.Audio;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.FilePointer;
import nl.colorize.multimedialib.renderer.GraphicsContext;
import nl.colorize.multimedialib.renderer.InputDevice;
import nl.colorize.multimedialib.renderer.InternetAccess;
import nl.colorize.multimedialib.renderer.KeyCode;
import nl.colorize.multimedialib.renderer.MediaLoader;
import nl.colorize.multimedialib.renderer.Renderer;
import nl.colorize.multimedialib.renderer.Updatable;
import nl.colorize.multimedialib.scene.Application;
import nl.colorize.multimedialib.scene.Effect;
import nl.colorize.multimedialib.scene.EffectManager;
import nl.colorize.multimedialib.scene.Scene;
import nl.colorize.multimedialib.scene.ui.Button;
import nl.colorize.multimedialib.scene.ui.Location;
import nl.colorize.multimedialib.scene.ui.SelectBox;
import nl.colorize.multimedialib.scene.ui.TextField;
import nl.colorize.multimedialib.scene.ui.WidgetStyle;
import nl.colorize.util.LogHelper;
import nl.colorize.util.animation.Interpolation;
import nl.colorize.util.animation.Timeline;
import nl.colorize.util.http.Headers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;

/**
 * Simple demo application that displays a number of animated Mario sprites on
 * top of a black background.
 * <p>
 * The demo application serves two purposes. First, it can be used as an example
 * application when using the framework to implement an application. Second, it
 * can be used for verification purposes to determine if a new platform is fully
 * supported.
 * <p>
 * The demo application can be started from the command line using the
 * {@link DemoLauncher}. It can also be embedded in applications by creating an
 * instance of this class from the application code.
 */
public class DemoApplication implements Scene {

    private Renderer renderer;
    private Application app;

    private SpriteSheet marioSpriteSheet;
    private TextFont font;
    private List<Mario> marios;
    private Audio audioClip;
    private EffectManager effectManager;
    private boolean canvasMask;
    private List<Scene> uiWidgets;

    public static final int DEFAULT_CANVAS_WIDTH = 800;
    public static final int DEFAULT_CANVAS_HEIGHT = 600;
    public static final int DEFAULT_FRAMERATE = 60;

    private static final FilePointer MARIO_SPRITES_FILE = new FilePointer("mario.png");
    private static final FilePointer AUDIO_FILE = new FilePointer("test.mp3");
    private static final FilePointer UI_WIDGET_FILE = new FilePointer("ui-widget-background.png");
    private static final FilePointer COLORIZE_LOGO = new FilePointer("colorize-logo.png");
    private static final int INITIAL_MARIOS = 20;
    private static final List<String> DIRECTIONS = ImmutableList.of("north", "east", "south", "west");
    private static final int NUM_BUTTONS = 7;
    private static final int BUTTON_WIDTH = 100;
    private static final int BUTTON_HEIGHT = 25;
    private static final ColorRGB RED_BUTTON = new ColorRGB(228, 93, 97);
    private static final ColorRGB GREEN_BUTTON = ColorRGB.parseHex("#72A725");
    private static final ColorRGB SHAPE_COLOR = new ColorRGB(200, 0, 0);
    private static final ColorRGB BACKGROUND_COLOR = ColorRGB.parseHex("#343434");
    private static final Transform MASK_TRANSFORM = Transform.withMask(ColorRGB.WHITE);
    private static final String TEST_URL = "http://www.colorize.nl";
    private static final Logger LOGGER = LogHelper.getLogger(DemoApplication.class);

    public DemoApplication(Application app) {
        this.app = app;
        this.renderer = app.getRenderer();
    }

    @Override
    public void start() {
        MediaLoader mediaLoader = renderer.getMediaLoader();

        initMarioSprites(mediaLoader);
        marios = new ArrayList<>();
        addMarios(INITIAL_MARIOS);

        font = mediaLoader.loadDefaultFont().derive(ColorRGB.WHITE);
        audioClip = mediaLoader.loadAudio(AUDIO_FILE);
        effectManager = new EffectManager();
        uiWidgets = Collections.emptyList();

        initEffects();
    }

    private void initMarioSprites(MediaLoader mediaLoader) {
        Image image = mediaLoader.loadImage(MARIO_SPRITES_FILE);
        marioSpriteSheet = new SpriteSheet(image);

        int y = 0;
        for (String direction : ImmutableList.of("north", "east", "south", "west")) {
            for (int i = 0; i <= 4; i++) {
                marioSpriteSheet.markRegion(direction + "_" + i, new Rect(i * 48, y, 48, 64));
            }
            y += 64;
        }
    }

    private void initEffects() {
        MediaLoader mediaLoader = app.getMediaLoader();

        Timeline animationTimeline = new Timeline(Interpolation.LINEAR, true);
        animationTimeline.addKeyFrame(0f, 0f);
        animationTimeline.addKeyFrame(2f, 1f);
        animationTimeline.addKeyFrame(4f, 0f);

        Image colorizeLogo = mediaLoader.loadImage(COLORIZE_LOGO);
        Effect effect = Effect.forImage(colorizeLogo, animationTimeline);
        Transform transform = effect.getTransform();
        effect.modify(value -> effect.setPosition(50, app.getCanvas().getHeight() - 50));
        effect.modify(value -> transform.setScale(80 + Math.round(value * 40f)));
        effect.modifyFrameUpdate(dt -> transform.addRotation(Math.round(dt * 100f)));
        effectManager.play(effect);
    }

    @Override
    public void update(float deltaTime) {
        InputDevice inputDevice = renderer.getInputDevice();
        handleClick(inputDevice);

        for (Mario mario : marios) {
            mario.update(deltaTime);
        }

        effectManager.update(deltaTime);
        uiWidgets.forEach(widget -> widget.update(deltaTime));

        if (inputDevice.isKeyReleased(KeyCode.U)) {
            sendRequest();
        }
    }

    private void handleClick(InputDevice inputDevice) {
        for (int i = 0; i <= NUM_BUTTONS; i++) {
            if (isButtonClicked(inputDevice, i)) {
                handleButtonClick(i);
                return;
            }
        }

        for (Mario mario : marios) {
            if (inputDevice.isPointerReleased(mario.getBounds())) {
                mario.mask = !mario.mask;
                return;
            }
        }

        if (inputDevice.isPointerReleased(app.getCanvas().getBounds())) {
            createTouchMarker(inputDevice.getPointers());
        }
    }

    private boolean isButtonClicked(InputDevice input, int buttonIndex) {
        Rect buttonBounds = new Rect(renderer.getCanvas().getWidth() - BUTTON_WIDTH,
            buttonIndex * 30, BUTTON_WIDTH, BUTTON_HEIGHT);
        return input.isPointerReleased(buttonBounds);
    }

    private void handleButtonClick(int index) {
        switch (index) {
            case 0 : addMarios(10); break;
            case 1 : removeMarios(10); break;
            case 2 : audioClip.play(); break;
            case 3 : canvasMask = !canvasMask; break;
            case 4 : initUIWidgets(); break;
            default : break;
        }
    }

    private void initUIWidgets() {
        if (!uiWidgets.isEmpty()) {
            uiWidgets = Collections.emptyList();
            return;
        }

        Image widget = app.getMediaLoader().loadImage(UI_WIDGET_FILE);

        Button button = new Button(new WidgetStyle(widget, font), "Click");
        button.setLocation(Location.fixed(200, 200));
        button.setClickHandler(app.getInputDevice(), () -> LOGGER.info("Button clicked"));

        SelectBox select = new SelectBox(new WidgetStyle(widget, font),
            ImmutableList.of("A", "B", "C"), "A");
        select.setLocation(Location.fixed(200, 240));
        select.setClickHandler(app.getInputDevice(), item -> LOGGER.info("Selected item " + item));

        TextField textField = new TextField(new WidgetStyle(widget, font), "Enter text:");
        textField.setLocation(Location.fixed(200, 280));
        textField.setChangeHandler(app.getInputDevice(), text -> LOGGER.info("Entered text: " + text));

        uiWidgets = ImmutableList.of(button, select, textField);
    }

    private void createTouchMarker(List<Point> positions) {
        for (Point position : positions) {
            Timeline timeline = new Timeline();
            timeline.addKeyFrame(0f, 100f);
            timeline.addKeyFrame(1f, 100f);
            timeline.addKeyFrame(1.5f, 0f);

            String text = Math.round(position.getX()) + ", " + Math.round(position.getY());

            Effect effect = Effect.forTextAlpha(text, font, Align.LEFT, timeline);
            effect.setPosition(position);
            effectManager.play(effect);
        }
    }

    @Override
    public void render(GraphicsContext graphics) {
        graphics.drawBackground(BACKGROUND_COLOR);
        drawSprites(graphics);
        drawHUD(graphics);
        if (canvasMask) {
            graphics.drawRect(new Rect(10f, 10f, DEFAULT_CANVAS_WIDTH - 20f, DEFAULT_CANVAS_HEIGHT - 20f),
                ColorRGB.WHITE, Transform.withAlpha(10));
        }
        effectManager.render(graphics);
        uiWidgets.forEach(widget -> widget.render(graphics));
    }

    private void drawSprites(GraphicsContext graphics) {
        for (Mario mario : marios) {
            Transform transform = mario.mask ? MASK_TRANSFORM : null;
            graphics.drawSprite(mario.sprite, mario.position.getX(), mario.position.getY(), transform);
        }
    }

    private void drawHUD(GraphicsContext graphics) {
        drawButton(graphics, "Add sprites", RED_BUTTON, 0);
        drawButton(graphics, "Remove sprites", RED_BUTTON, 30);
        drawButton(graphics, "Play sound", GREEN_BUTTON, 60);
        drawButton(graphics, "Canvas bounds", GREEN_BUTTON, 90);
        drawButton(graphics, "UI widgets", GREEN_BUTTON, 120);

        Canvas canvas = renderer.getCanvas();

        graphics.drawText("Canvas:  " + canvas, font, 20, 20);
        graphics.drawText("Framerate:  " + Math.round(app.getAverageFPS()), font, 20, 40);
        graphics.drawText("Frame time:  " + Math.round(app.getAverageFrameTime()) + "ms",
            font, 20, 60);
        graphics.drawText("Sprites:  " + marios.size(), font, 20, 80);
    }

    private void drawButton(GraphicsContext graphics, String label, ColorRGB background, int y) {
        graphics.drawRect(new Rect(renderer.getCanvas().getWidth() - BUTTON_WIDTH - 2, y + 2,
            BUTTON_WIDTH, BUTTON_HEIGHT), background, null);
        graphics.drawText(label, font, renderer.getCanvas().getWidth() - BUTTON_WIDTH / 2f, y + 17,
            Align.CENTER);
    }

    public void addMarios(int amount) {
        for (int i = 0; i < amount; i++) {
            Sprite marioSprite = createMarioSprite();
            marios.add(new Mario(marioSprite,
                new Rect(0, 0, renderer.getCanvas().getWidth(), renderer.getCanvas().getHeight())));
        }
    }

    private Sprite createMarioSprite() {
        Sprite marioSprite = new Sprite();
        for (String direction : DIRECTIONS) {
            List<Image> frames = marioSpriteSheet.get(direction + "_0",
                direction + "_1", direction + "_2", direction + "_3", direction + "_4");
            Animation anim = new Animation(frames, 0.1f, true);
            marioSprite.addState(direction, anim);
        }
        return marioSprite;
    }

    private void removeMarios(int amount) {
        for (int i = 0; i < amount && !marios.isEmpty(); i++) {
            marios.remove(marios.size() - 1);
        }
    }

    private void sendRequest() {
        LOGGER.info("Sending request to " + TEST_URL);

        InternetAccess internetAccess = app.getInternetAccess();
        Headers headers = new Headers();
        headers.add(HttpHeaders.X_DO_NOT_TRACK, "1");
        internetAccess.get(TEST_URL, headers)
            .then(LOGGER::info)
            .thenCatch(error -> LOGGER.warning("Sending request failed"));
    }

    /**
     * Represents one of the mario sprites that walks around the scene.
     */
    private static class Mario implements Updatable {

        private Sprite sprite;
        private Rect canvasBounds;
        private Point position;
        private int direction;
        private int speed;
        private boolean mask;

        public Mario(Sprite sprite, Rect canvasBounds) {
            this.sprite = sprite;
            this.position = new Point(RandomGenerator.getFloat(0f, canvasBounds.getWidth()),
                RandomGenerator.getFloat(0f, canvasBounds.getHeight()));
            this.canvasBounds = canvasBounds;
            this.direction = RandomGenerator.getInt(0, 4);
            this.speed = RandomGenerator.getInt(1, 4);
            this.mask = false;
        }

        @Override
        public void update(float deltaTime) {
            sprite.changeState(DIRECTIONS.get(direction));
            sprite.update(deltaTime);

            switch (direction) {
                case 0 : position.add(0, -speed); break;
                case 1 : position.add(speed, 0); break;
                case 2 : position.add(0, speed); break;
                case 3 : position.add(-speed, 0); break;
                default : throw new AssertionError();
            }

            checkBounds();
        }

        private void checkBounds() {
            if (position.getX() < 0 || position.getX() > canvasBounds.getWidth() ||
                position.getY() < 0 || position.getY() > canvasBounds.getHeight()) {
                direction = (direction + 2) % 4;
            }
        }

        private Rect getBounds() {
            return new Rect(position.getX() - sprite.getCurrentWidth() / 2f,
                position.getY() - sprite.getCurrentHeight() / 2f,
                sprite.getCurrentWidth(), sprite.getCurrentHeight());
        }
    }
}
