//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

let canvas = null;
let overlayCanvas = null;
let renderer = null;

let imageContainer = null;
let images = {};
let imageDataCache = {};
let audioContainer = null;
let audio = {};
let fontContainer = null;
let resourceContainer = null;

let pointerEventBuffer = [];
let keyStates = {};

let socket = null;

const WEBGL_ACTIVE = window.location.href.indexOf("webgl=true") != -1;
const WEBGL_SUPPORTED = window.WebGLRenderingContext != null && WEBGL_ACTIVE;
const CONSOLE_ENABLED = window.location.href.indexOf("console=true") != -1;

document.addEventListener("DOMContentLoaded", event => {
    imageContainer = document.getElementById("imageContainer");
    audioContainer = document.getElementById("audioContainer");
    fontContainer = document.getElementById("fontContainer");
    resourceContainer = document.getElementById("resourceContainer");

    let container = document.getElementById("multimediaLibContainer");
    initCanvas(container);
    main();
});

function initCanvas(container) {
    let width = Math.round(container.offsetWidth);
    let height = Math.round(document.documentElement.clientHeight);

    canvas = createCanvas(width, height);
    container.appendChild(canvas);

    renderer = createRenderer();
    initEventHandlers(container);

    if (renderer.hasOverlayCanvas()) {
        overlayCanvas = createCanvas(width, height);
        overlayCanvas.id = "overlayCanvas";
        container.appendChild(overlayCanvas);
    }

    document.getElementById("loading").style.display = "none";
}

function initEventHandlers(container) {
    canvas.addEventListener("mousedown", event => registerMouseEvent("mousedown", event), false);
    canvas.addEventListener("mouseup", event => registerMouseEvent("mouseup", event), false);
    canvas.addEventListener("mousemove", event => registerMouseEvent("mousemove", event), false);
    window.addEventListener("mouseout", event => registerMouseEvent("mouseout", event), false);
    canvas.addEventListener("touchstart", event => registerTouchEvent("touchstart", event), false);
    canvas.addEventListener("touchend", event => registerTouchEvent("touchend", event), false);
    canvas.addEventListener("touchmove", event => registerTouchEvent("touchmove", event), false);
    window.addEventListener("touchcancel", event => registerTouchEvent("touchcancel", event), false);
    window.addEventListener("keydown", onKeyDown);
    window.addEventListener("keyup", onKeyUp);
    window.addEventListener("resize", () => resizeCanvas(container));
}

function createCanvas(width, height) {
    let newCanvas = document.createElement("canvas");
    newCanvas.style.width = width + "px";
    newCanvas.style.height = height + "px";
    newCanvas.width = width * window.devicePixelRatio;
    newCanvas.height = height * window.devicePixelRatio;
    return newCanvas;
}

function createRenderer() {
    if (WEBGL_SUPPORTED) {
        let glContext = canvas.getContext("webgl");
        if (glContext != null) {
            console.log("Using WebGL renderer");
            return new WebGLRenderer(glContext);
        }
    }

    console.log("Using HTML5 canvas renderer");
    let context = canvas.getContext("2d");
    return new Html5CanvasRenderer(context);
}

function resizeCanvas(container) {
    let targetWidth = Math.round(container.offsetWidth);
    let targetHeight = Math.round(document.documentElement.clientHeight);

    canvas.style.width = targetWidth + "px";
    canvas.style.height = targetHeight + "px";
    canvas.width = targetWidth * window.devicePixelRatio;
    canvas.height = targetHeight * window.devicePixelRatio;

    if (overlayCanvas != null) {
        overlayCanvas.style.width = targetWidth + "px";
        overlayCanvas.style.height = targetHeight + "px";
        overlayCanvas.width = targetWidth * window.devicePixelRatio;
        overlayCanvas.height = targetHeight * window.devicePixelRatio;
    }
}

function onFrame(callback) {
    renderer.clearCanvas();
    callback();
    window.requestAnimationFrame(() => onFrame(callback));
}

function registerMouseEvent(eventType, event) {
    let mouseX = event.pageX - canvas.offsetLeft;
    let mouseY = event.pageY - canvas.offsetTop;
    registerPointerEvent(eventType + ";mouse;" + mouseX + ";" + mouseY);
    cancelEvent(event);
}

function registerTouchEvent(eventType, event) {
    for (let i = 0; i < event.changedTouches.length; i++) {
        let identifier = event.changedTouches[i].identifier;
        let touchX = event.changedTouches[i].pageX - canvas.offsetLeft;
        let touchY = event.changedTouches[i].pageY - canvas.offsetTop;
        registerPointerEvent(eventType + ";" + identifier + ";" + touchX + ";" + touchY);
    }
}

function registerPointerEvent(entry) {
    pointerEventBuffer.push(entry);
    if (CONSOLE_ENABLED && entry.indexOf("move") == -1) {
        logConsoleEvent(entry);
    }
}

function flushPointerEventBuffer() {
    let flushed = pointerEventBuffer;
    pointerEventBuffer = [];
    return flushed;
}

function onKeyDown(event) {
    keyStates[event.keyCode] = 1;
    cancelEvent(event);
}

function onKeyUp(event) {
    keyStates[event.keyCode] = 0;
    cancelEvent(event);
}

function cancelEvent(event) {
    event.preventDefault();
    event.stopPropagation();
}

function loadImage(id, path) {
    let imageElement = document.createElement("img");
    imageElement.onload = () => renderer.onLoadImage(id, imageElement);
    imageElement.src = path;
    imageContainer.appendChild(imageElement);
    images[id] = imageElement;
}

function loadAudio(id, path) {
    audio[id] = new Audio(path);
}

function loadFont(id, path, fontFamily) {
    let css = "";
    css += "@font-face { ";
    css += "    font-family: '" + fontFamily + "'; ";
    css += "    font-style: normal; ";
    css += "    font-weight: 400; ";
    css += "    src: url('" + path + "') format('truetype'); ";
    css += "}; ";

    let style = document.createElement("style");
    style.type = "text/css";
    style.appendChild(document.createTextNode(css));
    fontContainer.appendChild(style);
}

function getImageData(id, x, y) {
    if (images[id]) {
        let image = images[id];
        let imageData = imageDataCache[id];

        if (imageData == null) {
            let imageCanvas = createCanvas(image.width, image.height);
            let imageCanvasContext = imageCanvas.getContext("2d");
            imageCanvasContext.drawImage(image, 0, 0);
            imageData = imageCanvasContext;

            if (isImageDataAvailable(imageData, x, y)) {
                imageDataCache[id] = imageData;
            }
        }

        return imageData.getImageData(x, y, 1, 1).data;
    } else {
        return [-1, -1, -1, 255];
    }
}

function isImageDataAvailable(imageData, x, y) {
    let rgba = imageData.getImageData(x, y, 1, 1).data;
    return (rgba[0] + rgba[1] + rgba[2] + rgba[3]) != 0;
}

/**
 * Converts a hexadecimal color to an array of red, green, and blue components
 * represented by an integer between 0 and 255. For example, #FF0000 will return
 * the array [255, 0, 0].
 */
function toRGB(hexColor) {
    if (hexColor.length != 7) {
        throw "Invalid hexadecimal color: " + hexColor;
    }

    return [
        parseInt(hexColor.substring(1, 3), 16),
        parseInt(hexColor.substring(3, 5), 16),
        parseInt(hexColor.substring(5, 7), 16)
    ];
}

/**
 * Converts a hexadecimal color and alpha channel into an array with 4 elements,
 * each representd by a value between 0.0 and 1.0.
 */
function toRGBA(hexColor, alpha) {
    let rgb = toRGB(hexColor);
    return [rgb[0] / 255.0, rgb[1] / 255.0, rgb[2] / 255.0, alpha];
}

function playAudio(id, volume, loop) {
    audio[id].volume = volume;
    audio[id].loop = loop;
    audio[id].play();
}

function stopAudio(id, reset) {
    audio[id].pause();
    if (reset) {
        audio[id].currentTime = 0.0;
    }
}

function sendGetRequest(url, headers, callback) {
    let request = new XMLHttpRequest();
    request.onreadystatechange = () => {
        if (request.readyState == XMLHttpRequest.DONE) {
            callback(request.responseText);
        }
    };
    request.open("GET", url, true);
    prepareRequest(request, headers);
    request.send();
}

function sendPostRequest(url, headers, params, callback) {
    let request = new XMLHttpRequest();
    request.onreadystatechange = () => {
        if (request.readyState == XMLHttpRequest.DONE) {
            callback(request.responseText);
        }
    };
    request.open("POST", url, true);
    prepareRequest(request, headers);
    request.send(params);
}

function prepareRequest(request, headers) {
    request.setRequestHeader("X-Requested-With", "MultimediaLib");
    for (let i = 0; i < headers.length; i += 2) {
        request.setRequestHeader(headers[i], headers[i + 1]);
    }
}

function isWebSocketSupported() {
    return "WebSocket" in window;
}

function connectWebSocket(uri, callback) {
    socket = new WebSocket(uri);
    socket.onopen = event => callback("__open");
    socket.onmessage = event => callback(event.data);
    socket.onerror = error => console.log("Web socket error: " + error.message);
}

function sendWebSocket(message) {
    if (socket == null) {
        throw "Web socket not open";
    }

    socket.send(message);
}

function closeWebSocket() {
    if (socket != null) {
        socket.close();
        socket = null;
    }
}

function logConsoleEvent(message) {
    console.log(message);

    if (CONSOLE_ENABLED) {
        let browserConsole = document.getElementById("console");
        browserConsole.innerHTML = "<div>" + message + "</div>" + browserConsole.innerHTML;
    }
}

function takeScreenshot() {
    if (canvas == null) {
        throw "Canvas not yet initialized";
    }
    return canvas.toDataURL();
}

/**
 * Renderer implementation that uses the HTML5 canvas drawing API. Whether drawing
 * operations are hardware accelerated depends on the browser and on the platform.
 */
class Html5CanvasRenderer {

    constructor(context) {
        this.context = context;
        this.maskCache = {};
    }

    hasOverlayCanvas() {
        return false;
    }

    onLoadImage(id, imageElement) {
    }

    clearCanvas() {
        this.context.clearRect(0, 0, canvas.width, canvas.height);
    }

    drawRect(x, y, width, height, color, alpha) {
        this.context.globalAlpha = alpha;
        this.context.fillStyle = color;
        this.context.fillRect(x, y, width, height);
        this.context.globalAlpha = 1.0;
    }

    drawCircle(x, y, radius, color, alpha) {
        this.context.globalAlpha = alpha;
        this.context.fillStyle = color;
        this.context.beginPath();
        this.context.arc(x, y, radius, 0, 2.0 * Math.PI);
        this.context.fill();
        this.context.globalAlpha = 1.0;
    }

    drawPolygon(points, color, alpha) {
        this.context.fillStyle = color;
        this.context.globalAlpha = alpha;
        this.context.beginPath();
        this.context.moveTo(points[0], points[1]);
        for (let i = 2; i < points.length; i += 2) {
            this.context.lineTo(points[i], points[i + 1]);
        }
        this.context.fill();
        this.context.globalAlpha = 1.0;
    }

    drawImage(id, x, y, width, height, alpha, mask) {
        if (images[id]) {
            this.drawImageRegion(id, 0, 0, images[id].width, images[id].height, x, y,
                width, height, 0.0, 1.0, 1.0, alpha, mask);
        }
    }

    drawImageRegion(id, regionX, regionY, regionWidth, regionHeight, x, y, width, height,
                    rotation, scaleX, scaleY, alpha, mask) {
        if (images[id]) {
            let image = this.prepareImage(images[id], mask);

            this.context.save();
            this.context.globalAlpha = alpha;
            this.context.translate(x, y);
            this.context.rotate(rotation);
            this.context.scale(scaleX, scaleY);
            this.context.drawImage(image, regionX, regionY, regionWidth, regionHeight,
                -width / 2.0, -height / 2.0, width, height);
            this.context.globalAlpha = 1.0;
            this.context.restore();
        }
    }

    prepareImage(image, mask) {
        if (!mask) {
            return image;
        }

        let cacheKey = image.width + "x" + image.height;
        let maskImageCanvas = this.maskCache[cacheKey];

        if (maskImageCanvas == null) {
            maskImageCanvas = createCanvas(image.width, image.height);
            this.maskCache[cacheKey] = maskImageCanvas;
        }

        let maskImageContext = maskImageCanvas.getContext("2d");
        maskImageContext.drawImage(image, 0, 0, image.width, image.height);
        maskImageContext.globalCompositeOperation = "source-atop";
        maskImageContext.fillStyle = mask;
        maskImageContext.fillRect(0, 0, image.width, image.height);

        return maskImageCanvas;
    }

    drawText(text, font, size, color, bold, x, y, align, alpha) {
        this.context.globalAlpha = alpha;
        this.context.fillStyle = color;
        this.context.font = (bold ? "bold " : "") + size + "px " + font;
        this.context.textAlign = align;
        this.context.fillText(text, x, y);
        this.context.globalAlpha = 1.0;
    }
}

/**
 * Uses WebGL to draw graphics on the canvas. WebGL is supported by all modern
 * browsers, but may not be supported by the platform itself.
 */
class WebGLRenderer {

    constructor(glContext) {
        if (glContext == null) {
            throw "WebGL not supported";
        }

        this.gl = glContext;
        this.textures = {};

        this.gl.enable(this.gl.BLEND);
        this.gl.blendFunc(this.gl.ONE, this.gl.ONE_MINUS_SRC_ALPHA);

        this.shaderProgram = this.initShaderProgram();

        let buffer = this.gl.createBuffer();
        this.gl.bindBuffer(this.gl.ARRAY_BUFFER, buffer);
    }

    initShaderProgram() {
        let vertexShader = this.loadShader(this.gl.VERTEX_SHADER, this.initVertexShader());
        let fragmentShader = this.loadShader(this.gl.FRAGMENT_SHADER, this.initFragmentShader());

        let shaderProgram = this.gl.createProgram();
        this.gl.attachShader(shaderProgram, vertexShader);
        this.gl.attachShader(shaderProgram, fragmentShader);
        this.gl.linkProgram(shaderProgram);

        if (!this.gl.getProgramParameter(shaderProgram, this.gl.LINK_STATUS)) {
            throw "Error while initializing WebGL shader: " + this.gl.getProgramInfoLog(shaderProgram);
        }

        return shaderProgram;
    }

    loadShader(type, glsl) {
        let shader = this.gl.createShader(type);
        this.gl.shaderSource(shader, glsl);
        this.gl.compileShader(shader);

        if (!this.gl.getShaderParameter(shader, this.gl.COMPILE_STATUS)) {
            throw "Error while loading WebGL shader: " + this.gl.getShaderInfoLog(shader);
        }

        return shader;
    }

    initVertexShader() {
        return `
            attribute vec4 aPosition;
            attribute vec4 aColor;
            attribute vec2 aTextureCoordinates;
            
            varying vec4 vColor;
            varying vec2 vTextureCoordinates;
            
            uniform vec2 uRotation;
            uniform vec2 uScale;

            void main() {
                vColor = aColor;
                vTextureCoordinates = aTextureCoordinates;
                
                vec2 rotatedPosition = vec2(
                    aPosition.x * uRotation.y + aPosition.y * uRotation.x,
                    aPosition.y * uRotation.y - aPosition.x * uRotation.x
                );

                gl_Position = vec4(rotatedPosition * uScale, 0.0, 1.0);
            }
        `;
    }

    initFragmentShader() {
        return `
            precision mediump float;

            uniform sampler2D uTextureUnit;
            uniform vec4 uColor;
            uniform float uAlpha;
            
            varying vec2 vTextureCoordinates;
            
            void main() {
                if (vTextureCoordinates.x >= 0.0 && vTextureCoordinates.y >= 0.0) {
                    vec4 color = texture2D(uTextureUnit, vTextureCoordinates);
                    gl_FragColor = vec4(color.rgb, uAlpha * color.a);
                } else {
                    gl_FragColor = uColor;
                }
            }
        `;
    }

    hasOverlayCanvas() {
        return true;
    }

    onLoadImage(id, imageElement) {
        let texture = this.loadTexture(id);
        this.gl.bindTexture(this.gl.TEXTURE_2D, texture);
        this.gl.texImage2D(this.gl.TEXTURE_2D, 0, this.gl.RGBA, this.gl.RGBA,
            this.gl.UNSIGNED_BYTE, imageElement);
    }

    clearCanvas() {
        this.gl.viewport(0, 0, this.gl.canvas.width, this.gl.canvas.height);
        this.gl.clearColor(0.0, 0.0, 0.0, 1.0);
        this.gl.clear(this.gl.COLOR_BUFFER_BIT | this.gl.DEPTH_BUFFER_BIT);

        this.initOverlayContext();
        this.overlayContext.clearRect(0, 0, overlayCanvas.width, overlayCanvas.height);

        this.gl.useProgram(this.shaderProgram);
    }

    drawRect(x, y, width, height, color, alpha) {
        let textureCoordinates = {
            s0: -1,
            s1: -1,
            t0: -1,
            t1: -1
        };

        let vertexData = this.toVertexData(x, y, width, height, textureCoordinates);
        this.renderPolygon(vertexData, null, toRGBA(color, alpha), [0.0, 0.0], [1.0, 1.0]);
    }

    drawCircle(x, y, radius, color, alpha) {
        //TODO
        console.log("drawCircle not yet implemented in WebGL renderer");
    }

    drawPolygon(points, color, alpha) {
        //TODO
        console.log("drawPolygon not yet implemented in WebGL renderer");
    }

    drawImage(id, x, y, width, height, alpha, mask) {
        if (images[id]) {
            this.drawImageRegion(id, 0, 0, images[id].width, images[id].height,
                x, y, width, height, 0, 100, 100, alpha, mask);
        }
    }

    drawImageRegion(id, regionX, regionY, regionWidth, regionHeight, x, y, width, height,
                    rotation, scaleX, scaleY, alpha, mask) {
        if (images[id]) {
            let textureCoordinates = {
                s0: regionX / images[id].width,
                s1: (regionX + regionWidth) / images[id].width,
                t0: 1.0 - ((regionY + regionHeight) / images[id].height),
                t1: 1.0 - (regionY / images[id].height),
            };

            let vertexData = this.toVertexData(x - width / 2, y - height / 2, width, height,
                textureCoordinates);
            let texture = this.loadTexture(id);
            let color = toRGBA("#FFFFFF", alpha);
            let rotationVector = [Math.sin(rotation), Math.cos(rotation)];
            let scaleVector = [1.0, 1.0];

            this.renderPolygon(vertexData, texture, color, rotationVector, scaleVector);
        }
    }

    drawText(text, font, size, color, bold, x, y, align, alpha) {
        this.initOverlayContext();

        this.overlayContext.globalAlpha = alpha;
        this.overlayContext.fillStyle = color;
        this.overlayContext.font = (bold ? "bold " : "") + size + "px " + font;
        this.overlayContext.textAlign = align;
        this.overlayContext.fillText(text, x, y);
        this.overlayContext.globalAlpha = 1.0;
    }

    initOverlayContext() {
        if (this.overlayContext == null) {
            this.overlayContext = overlayCanvas.getContext("2d");
        }
    }

    toVertexData(x, y, width, height, textureCoordinates) {
        let centerS = textureCoordinates.s0 + (textureCoordinates.s1 - textureCoordinates.s0) / 2.0;
        let centerT = textureCoordinates.t1 + (textureCoordinates.t0 - textureCoordinates.t1) / 2.0;

        let vertexData = [
            x + width / 2, y + height / 2, centerS, centerT,
            x, y + height, textureCoordinates.s0, textureCoordinates.t1,
            x + width, y + height, textureCoordinates.s1, textureCoordinates.t1,
            x + width, y, textureCoordinates.s1, textureCoordinates.t0,
            x, y, textureCoordinates.s0, textureCoordinates.t0,
            x, y + height, textureCoordinates.s0, textureCoordinates.t1
        ];

        // Converts X and Y coordinates in an array of vertices from the canvas
        // coordinate space (0, 0, width, height) to the OpenGL ES coordinate
        // space (-1, 1, 1, -1).
        for (let i = 0; i < vertexData.length; i += 4) {
            vertexData[i] = (vertexData[i] / canvas.width * 2.0) - 1.0;
            vertexData[i + 1] = ((vertexData[i + 1] / canvas.height * 2.0) - 1.0) * -1.0;
        }

        return vertexData;
    }

    loadTexture(id) {
        if (this.textures[id]) {
            return this.textures[id];
        }

        let texture = this.gl.createTexture();
        this.gl.bindTexture(this.gl.TEXTURE_2D, texture);
        this.gl.texImage2D(this.gl.TEXTURE_2D, 0, this.gl.RGBA, 1, 1, 0, this.gl.RGBA,
            this.gl.UNSIGNED_BYTE, new Uint8Array([255, 255, 255, 255]));
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_WRAP_S, this.gl.CLAMP_TO_EDGE);
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_WRAP_T, this.gl.CLAMP_TO_EDGE);
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_MIN_FILTER, this.gl.LINEAR);
        this.gl.texParameteri(this.gl.TEXTURE_2D, this.gl.TEXTURE_MAG_FILTER, this.gl.LINEAR);
        this.textures[id] = texture;
        return texture;
    }

    renderPolygon(vertexData, texture, colorVector, rotationVector, scaleVector) {
        let positionLocation = this.gl.getAttribLocation(this.shaderProgram, "aPosition");
        let rotationLocation = this.gl.getUniformLocation(this.shaderProgram, "uRotation");
        let scaleLocation = this.gl.getUniformLocation(this.shaderProgram, "uScale");
        let colorLocation = this.gl.getUniformLocation(this.shaderProgram, "uColor");
        let alphaLocation = this.gl.getUniformLocation(this.shaderProgram, "uAlpha");
        let textureCoordsLocation = this.gl.getAttribLocation(this.shaderProgram, "aTextureCoordinates");

        this.gl.uniform2fv(rotationLocation, rotationVector);
        this.gl.uniform2fv(scaleLocation, scaleVector);
        this.gl.uniform4fv(colorLocation, colorVector);
        this.gl.uniform1f(alphaLocation, colorVector[3]);
        this.gl.bufferData(this.gl.ARRAY_BUFFER, new Float32Array(vertexData), this.gl.STATIC_DRAW);

        if (texture != null) {
            this.gl.bindTexture(this.gl.TEXTURE_2D, texture);
        } else {
            this.gl.bindTexture(this.gl.TEXTURE_2D, null);
        }

        this.gl.vertexAttribPointer(positionLocation, 2, this.gl.FLOAT, false, 16, 0);
        this.gl.enableVertexAttribArray(positionLocation);
        this.gl.vertexAttribPointer(textureCoordsLocation, 2, this.gl.FLOAT, false, 16, 8);
        this.gl.enableVertexAttribArray(textureCoordsLocation);

        this.gl.drawArrays(this.gl.TRIANGLE_FAN, 0, 6);

        this.gl.disableVertexAttribArray(positionLocation);
        this.gl.disableVertexAttribArray(textureCoordsLocation);
    }
}
