//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.renderer.java2d;

import com.google.common.annotations.VisibleForTesting;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.GraphicsMode;
import nl.colorize.multimedialib.renderer.MediaException;
import nl.colorize.multimedialib.renderer.NestedRenderCallback;
import nl.colorize.util.swing.Utils2D;

import javax.swing.JFrame;
import java.awt.BufferCapabilities;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferStrategy;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

/**
 * Headless implementation of a {@link Java2DRenderer}. This is primarily
 * intended for testing purposes in environments where graphics are not
 * available. Graphics are displayed on an image instead.
 */
@VisibleForTesting
public class HeadlessRenderer extends Java2DRenderer {

    private HeadlessWindowBuffer buffer;

    public HeadlessRenderer(Canvas canvas, int framerate, WindowOptions options) {
        super(canvas, framerate, options);
    }

    @Override
    public GraphicsMode getSupportedGraphicsMode() {
        return GraphicsMode.HEADLESS;
    }

    @Override
    protected JFrame initializeWindow(WindowOptions windowOptions) {
        return null;
    }

    @Override
    protected void prepareCanvas() {
    }

    @Override
    public void drawFrame(Graphics2D g2) {
        super.drawFrame(g2);
    }

    @Override
    protected BufferStrategy prepareWindowBuffer() {
        Canvas canvas = getCanvas();

        if (isBufferDirty(canvas)) {
            buffer = new HeadlessWindowBuffer();
            buffer.image = new BufferedImage(canvas.getWidth(), canvas.getHeight(),
                BufferedImage.TYPE_INT_ARGB);
        }

        return buffer;
    }

    private boolean isBufferDirty(Canvas canvas) {
        if (buffer == null || buffer.image == null) {
            return true;
        }

        return buffer.image.getWidth() != canvas.getWidth() ||
            buffer.image.getHeight() != canvas.getHeight();
    }

    @Override
    protected void blitGraphicsContext(BufferStrategy windowBuffer) {
    }

    @Override
    public NestedRenderCallback getCallbacks() {
        return super.getCallbacks();
    }

    public BufferedImage getImage() {
        return buffer.image;
    }

    public void saveScreenshot(File dest) {
        try {
            Utils2D.savePNG(buffer.image, dest);
        } catch (IOException e) {
            throw new MediaException("Cannot save image", e);
        }
    }

    private static class HeadlessWindowBuffer extends BufferStrategy {

        private BufferedImage image;

        @Override
        public BufferCapabilities getCapabilities() {
            return null;
        }

        @Override
        public Graphics getDrawGraphics() {
            return Utils2D.createGraphics(image, true, true);
        }

        @Override
        public boolean contentsLost() {
            return false;
        }

        @Override
        public boolean contentsRestored() {
            return false;
        }

        @Override
        public void show() {
        }
    }
}
