//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.scene;

import com.google.common.base.Preconditions;
import nl.colorize.multimedialib.renderer.Updatable;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * Logic that is active for a certain amount of time during a scene. Actions
 * can be used to declare scene behavior in a declarative way. Behavior can
 * be added when the action is started, when it completes, for every frame
 * while it is active. Actions can be configured to remain active for a certain
 * amount of time, until a certain condition is reached, or indefinitely for
 * the remainder of the scene.
 * <p>
 * The administration of playing actions is usually handled in combination
 * with a {@link ActionManager} that keeps track of all actions that are played
 * during a scene.
 * <p>
 * Actions contain only logic and do not define graphical behavior. Use
 * {@link Effect}s to have a similar mechanism for defining graphics that
 * should be shown during the scene.
 */
public class Action implements Updatable {

    private Predicate<Action> terminator;
    private List<Runnable> startCallbacks;
    private List<Consumer<Float>> frameCallbacks;
    private List<Runnable> completedCallbacks;
    private Action chained;

    private float time;
    private boolean started;
    private boolean completed;

    protected Action(Predicate<Action> terminator) {
        this.terminator = terminator;
        this.startCallbacks = new ArrayList<>();
        this.frameCallbacks = new ArrayList<>();
        this.completedCallbacks = new ArrayList<>();

        this.time = 0f;
        this.started = false;
        this.completed = false;
    }

    public Action onStart(Runnable callback) {
        startCallbacks.add(callback);
        return this;
    }

    public Action onFrame(Consumer<Float> callback) {
        frameCallbacks.add(callback);
        return this;
    }

    public Action onFrame(Runnable callback) {
        return onFrame(deltaTime -> callback.run());
    }

    public Action onCompleted(Runnable callback) {
        completedCallbacks.add(callback);
        return this;
    }

    /**
     * Chains another action that should be performed immediately after this
     * action has been completed.
     *
     * @throws IllegalStateException if another chained action has already
     *         been registered.
     */
    public Action chain(Action next) {
        Preconditions.checkState(chained == null, "Chained action already defined");
        
        chained = next;
        return this;
    }

    @Override
    public void update(float deltaTime) {
        if (!started) {
            started = true;
            startCallbacks.forEach(Runnable::run);
        }

        if (completed) {
            if (chained != null) {
                chained.update(deltaTime);
            }
        } else {
            time += deltaTime;
            frameCallbacks.forEach(callback -> callback.accept(deltaTime));

            if (terminator.test(this)) {
                completed = true;
                completedCallbacks.forEach(Runnable::run);
            }
        }
    }

    public boolean isCompleted() {
        return completed && (chained == null || chained.isCompleted());
    }

    public boolean isActive() {
        return started && !isCompleted();
    }

    /**
     * Creates an action that will remain active indefinitely.
     */
    public static Action indefinitely() {
        return until(action -> false);
    }

    /**
     * Creates an action that will remain active for a limited amount of time
     * @param duration Action duration in seconds.
     */
    public static Action timed(float duration) {
        Preconditions.checkArgument(duration > 0f, "Invalid duration: " + duration);

        return until(action -> action.time >= duration);
    }

    /**
     * Creates an action that will remain active until the specified condition
     * starts to return true.
     */
    public static Action until(Predicate<Action> terminator) {
        return new Action(terminator);
    }
}
