//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.renderer.java2d;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableList;
import com.google.common.net.HttpHeaders;
import nl.colorize.multimedialib.renderer.InternetAccess;
import nl.colorize.multimedialib.renderer.WebSocketConnection;
import nl.colorize.util.LogHelper;
import nl.colorize.util.Promise;
import nl.colorize.util.http.Headers;
import nl.colorize.util.http.Method;
import nl.colorize.util.http.PostData;
import nl.colorize.util.http.URLLoader;
import org.java_websocket.client.WebSocketClient;
import org.java_websocket.handshake.ServerHandshake;

import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Sends HTTP requests using the HTTP client included as part of the Java
 * standard library. It also provides web socket support through the
 * Java-WebSocket library.
 */
public class StandardInternetAccess implements InternetAccess {

    private static final Logger LOGGER = LogHelper.getLogger(StandardInternetAccess.class);

    @Override
    public Promise<String> get(String url, Headers headers) {
        return send(Method.GET, url, headers, null);
    }

    @Override
    public Promise<String> post(String url, Headers headers, PostData body) {
        return send(Method.POST, url, headers, body);
    }

    public Promise<String> send(Method method, String url, Headers headers, PostData body) {
        URLLoader request = URLLoader.create(method, url, StandardCharsets.UTF_8);
        request.addHeader(HttpHeaders.X_REQUESTED_WITH, "MultimediaLib");
        if (headers != null) {
            request.addHeaders(headers);
        }

        if (body != null) {
            request.setBody(body);
        }

        return new Promise<String>(promise -> {
            request.sendRequestPromise()
                .then(response -> promise.resolve(response.getBody()))
                .thenCatch(promise::reject);
        });
    }

    @Override
    public boolean isWebSocketSupported() {
        return true;
    }

    @Override
    public WebSocketConnection connectWebSocket(String uri) {
        JavaWebSocketClient client = new JavaWebSocketClient(URI.create(uri));
        client.connect();
        return client;
    }

    /**
     * Implementation of web socket client using the Java-WebSocket library.
     * Note that the client will run in a separate thread, which is different
     * from the renderer thread.
     */
    private static class JavaWebSocketClient extends WebSocketClient implements WebSocketConnection {

        private AtomicBoolean connected;
        private List<String> sendBuffer;
        private List<String> receiveBuffer;

        public JavaWebSocketClient(URI uri) {
            super(uri);
            this.connected = new AtomicBoolean(false);
            this.sendBuffer = new CopyOnWriteArrayList<>();
            this.receiveBuffer = new CopyOnWriteArrayList<>();
        }

        @Override
        public void onOpen(ServerHandshake handshake) {
            LOGGER.info("Web socket connection: " + handshake.getHttpStatusMessage());
            connected.set(true);
            super.send("__init");
            sendBuffer.forEach(super::send);
            sendBuffer.clear();
        }

        @Override
        public void onClose(int code, String reason, boolean remote) {
            LOGGER.info("Web socket connection closed");
            connected.set(false);
        }

        @Override
        public void onMessage(String message) {
            receiveBuffer.add(message);
        }

        @Override
        public void onMessage(ByteBuffer buffer) {
            String message = new String(buffer.array(), Charsets.UTF_8);
            receiveBuffer.add(message);
        }

        @Override
        public void onError(Exception e) {
            LOGGER.log(Level.WARNING, "Web socket error", e);
        }

        @Override
        public void send(String message) {
            if (connected.get()) {
                super.send(message);
            } else {
                sendBuffer.add(message);
            }
        }

        @Override
        public List<String> flushReceivedMessages() {
            List<String> received = ImmutableList.copyOf(receiveBuffer);
            receiveBuffer.clear();
            return received;
        }
    }
}
