//-----------------------------------------------------------------------------
// Colorize MultimediaLib
// Copyright 2009-2020 Colorize
// Apache license (http://www.apache.org/licenses/LICENSE-2.0)
//-----------------------------------------------------------------------------

package nl.colorize.multimedialib.renderer.libgdx;

import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.graphics.g2d.SpriteBatch;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import nl.colorize.multimedialib.graphics.Align;
import nl.colorize.multimedialib.graphics.AlphaTransform;
import nl.colorize.multimedialib.graphics.ColorRGB;
import nl.colorize.multimedialib.graphics.Image;
import nl.colorize.multimedialib.graphics.TTFont;
import nl.colorize.multimedialib.graphics.Transform;
import nl.colorize.multimedialib.math.Circle;
import nl.colorize.multimedialib.math.Polygon;
import nl.colorize.multimedialib.math.Rect;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.GraphicsContext2D;

public class GDXGraphics2D implements GraphicsContext2D {

    private Canvas canvas;
    private SpriteBatch batch;
    private GDXMediaLoader mediaLoader;

    private static final int CIRCLE_PRECISION = 16;
    private static final Transform DEFAULT_TRANSFORM = new Transform();

    protected GDXGraphics2D(Canvas canvas, SpriteBatch batch, GDXMediaLoader mediaLoader) {
        this.canvas = canvas;
        this.batch = batch;
        this.mediaLoader = mediaLoader;
    }

    @Override
    public Canvas getCanvas() {
        return canvas;
    }

    @Override
    public void drawBackground(ColorRGB backgroundColor) {
        Canvas canvas = getCanvas();
        Rect background = new Rect(0f, 0f, canvas.getWidth(), canvas.getHeight());
        drawRect(background, backgroundColor, null);
    }

    @Override
    public void drawRect(Rect rect, ColorRGB color, AlphaTransform alpha) {
        Transform transform = new Transform();
        if (alpha != null) {
            transform.setAlpha(alpha.getAlpha());
        }

        Texture colorTexture = mediaLoader.getColorTexture(color);
        TextureRegion colorTextureRegion = new TextureRegion(colorTexture, 0, 0,
            colorTexture.getWidth(), colorTexture.getHeight());

        draw(rect.getCenterX(), rect.getCenterY(), rect.getWidth(), rect.getHeight(),
            color, colorTextureRegion, transform);
    }

    @Override
    public void drawCircle(Circle circle, ColorRGB color, AlphaTransform transform) {
        Polygon polygon = Polygon.createCircle(circle.getCenter(), circle.getRadius(), CIRCLE_PRECISION);
        drawPolygon(polygon, color, transform);
    }

    @Override
    public void drawPolygon(Polygon polygon, ColorRGB color, AlphaTransform alpha) {
        //TODO
        throw new UnsupportedOperationException();
    }

    @Override
    public void drawImage(Image image, float x, float y, Transform transform) {
        int width = image.getWidth();
        int height = image.getHeight();

        TextureRegion textureRegion = ((GDXImage) image).getTextureRegion();

        if (transform == null) {
            transform = DEFAULT_TRANSFORM;
        }

        draw(x, y, width, height, ColorRGB.WHITE, textureRegion, transform);
    }

    private void draw(float x, float y, float width, float height, ColorRGB color,
                      TextureRegion texture, Transform transform) {
        float canvasX = x - width / 2f;
        float canvasY = Math.abs(y - getCanvas().getHeight()) - height / 2f;

        float originX = width / 2f;
        float originY = height / 2f;

        batch.setColor(color.getR() / 255f, color.getG() / 255f, color.getB() / 255f,
            transform.getAlpha() / 100f);

        batch.draw(texture, canvasX, canvasY, originX, originY, width, height,
            transform.getScaleX() / 100f, transform.getScaleY() / 100f,
            transform.getRotation());
    }

    @Override
    public void drawText(String text, TTFont font, float x, float y, Align align, AlphaTransform transform) {
        BitmapFont bitmapFont = mediaLoader.getBitmapFont(font);
        float canvasY = Math.abs(y - getCanvas().getHeight());
        float baseline = font.getSize() * 0.3f;
        bitmapFont.draw(batch, text, x, canvasY + baseline, 0, getTextAlign(align), false);
    }

    private int getTextAlign(Align align) {
        switch (align) {
            case LEFT : return com.badlogic.gdx.utils.Align.left;
            case CENTER : return com.badlogic.gdx.utils.Align.center;
            case RIGHT : return com.badlogic.gdx.utils.Align.right;
            default : throw new AssertionError();
        }
    }
}
