/*
 * Decompiled with CFR 0.152.
 */
package nl.colorize.multimedialib.tool;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import nl.colorize.multimedialib.renderer.MediaException;
import nl.colorize.util.FileUtils;
import nl.colorize.util.Formatting;
import nl.colorize.util.LoadUtils;
import nl.colorize.util.LogHelper;
import nl.colorize.util.ResourceFile;
import nl.colorize.util.cli.CommandLineArgumentParser;
import nl.colorize.util.http.URLLoader;
import nl.colorize.util.http.URLResponse;
import nl.colorize.util.xml.XMLHelper;
import org.jdom2.Element;
import org.teavm.diagnostics.Problem;
import org.teavm.tooling.ConsoleTeaVMToolLog;
import org.teavm.tooling.TeaVMTargetType;
import org.teavm.tooling.TeaVMTool;
import org.teavm.tooling.TeaVMToolException;
import org.teavm.tooling.TeaVMToolLog;

public class TeaVMTranspilerTool {
    protected String projectName;
    protected File resourceDir;
    protected String mainClassName;
    protected File outputDir;
    protected boolean minify;
    protected File manifestFile;
    private static final List<String> FRAMEWORK_RESOURCE_FILE_PATHS = ImmutableList.of((Object)"browser/index.html", (Object)"browser/multimedialib.js", (Object)"browser/pixi-renderer.js", (Object)"browser/service-worker.js", (Object)"browser/assets/multimedialib.css", (Object)"browser/assets/favicon.png", (Object)"browser/assets/apple-icon.png", (Object)"browser/assets/orientation-lock.png", (Object)"browser/assets/loading.gif", (Object)"browser/assets/OpenSans-Regular.ttf", (Object)"colorize-logo.png", (Object)"orientation-lock.png", (Object[])new String[]{"ui-widget-background.png", "transition-effect.png"});
    private static final Map<String, String> NPM_VERSIONS = Maps.fromProperties((Properties)LoadUtils.loadProperties((ResourceFile)new ResourceFile("npm-versions.properties"), (Charset)Charsets.UTF_8));
    private static final List<String> JAVASCRIPT_URLS = ImmutableList.of((Object)("https://pixijs.download/v" + NPM_VERSIONS.get("pixi.js") + "/pixi.min.js"), (Object)("https://pixijs.download/v" + NPM_VERSIONS.get("pixi.js") + "/pixi.min.js.map"), (Object)("https://cdn.jsdelivr.net/npm/pixi-heaven@" + NPM_VERSIONS.get("pixi-heaven") + "/dist/pixi-heaven.umd.min.js"), (Object)("https://cdn.jsdelivr.net/npm/pixi-heaven@" + NPM_VERSIONS.get("pixi-heaven") + "/dist/pixi-heaven.umd.min.js.map"), (Object)("https://unpkg.com/three@" + NPM_VERSIONS.get("three") + "/build/three.min.js"), (Object)("https://unpkg.com/three@" + NPM_VERSIONS.get("three") + "/examples/js/loaders/GLTFLoader.js"), (Object)("https://unpkg.com/peerjs@" + NPM_VERSIONS.get("peerjs") + "/dist/peerjs.min.js"), (Object)("https://unpkg.com/peerjs@" + NPM_VERSIONS.get("peerjs") + "/dist/peerjs.min.cjs.map"));
    private static final List<String> TEXT_FILE_TYPES = ImmutableList.of((Object)".txt", (Object)".md", (Object)".json", (Object)".yml", (Object)".yaml", (Object)".properties", (Object)".fnt", (Object)".csv", (Object)"-manifest");
    private static final List<String> RESOURCE_FILE_TYPES = ImmutableList.of((Object)".css", (Object)".png", (Object)".jpg", (Object)".svg", (Object)".gif", (Object)".ttf", (Object)".wav", (Object)".mp3", (Object)".ogg", (Object)".gltf", (Object)".fbx", (Object)".g3db", (Object[])new String[0]);
    private static final List<String> KNOWN_MISSING_CLASSES = ImmutableList.of((Object)"[java.lang.System.exit(I)V]", (Object)"[java.lang.reflect.TypeVariable]", (Object)"[java.lang.Class.getGenericSuperclass()Ljava/lang/reflect/Type;]");
    private static final List<String> SUPPORTED_RENDERERS = ImmutableList.of((Object)"canvas", (Object)"pixi", (Object)"three");
    private static final Logger LOGGER = LogHelper.getLogger(TeaVMTranspilerTool.class);

    public static void main(String[] args) {
        CommandLineArgumentParser argParser = new CommandLineArgumentParser("TeaVMTranspiler").add("--project", "Project name for the application").add("--main", "Main class that acts as application entry point").add("--resources", "Location of the application's resource files").add("--out", "Output directory for the generated files").addOptional("--manifest", null, "PWA manifest.json file location").addFlag("--minify", "Minifies the generated JavaScript, off by default");
        argParser.parseArgs(args);
        TeaVMTranspilerTool tool = new TeaVMTranspilerTool();
        tool.projectName = argParser.get("project");
        tool.mainClassName = argParser.get("main");
        tool.resourceDir = argParser.getDir("resources");
        tool.manifestFile = argParser.getFile("manifest");
        tool.outputDir = argParser.getFile("out");
        tool.minify = argParser.getBool("minify");
        tool.run();
    }

    protected void run() {
        Preconditions.checkArgument((boolean)this.resourceDir.exists(), (Object)("Resource directory not found: " + this.resourceDir.getAbsolutePath()));
        this.outputDir.mkdir();
        this.checkMainClass();
        try {
            this.copyResources();
            JAVASCRIPT_URLS.forEach(this::downloadJavaScriptLibrary);
            this.transpile();
            if (this.manifestFile != null) {
                this.copyManifest();
                this.generateServiceWorker();
            }
            this.printSummary();
        }
        catch (TeaVMToolException e) {
            LOGGER.log(Level.SEVERE, "Transpiling failed", e);
        }
    }

    private void checkMainClass() {
        try {
            Class<?> mainClass = Class.forName(this.mainClassName);
            mainClass.getDeclaredMethod("main", String[].class);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Invalid main class: " + this.mainClassName, e);
        }
    }

    private void printSummary() {
        long htmlSize = new File(this.outputDir, "index.html").length();
        long jsSize = new File(this.outputDir, "classes.js").length();
        long libSize = Arrays.stream(new File(this.outputDir, "libraries").listFiles()).mapToLong(File::length).sum();
        LOGGER.info("HTML file size:                     " + Formatting.memoryFormat((long)htmlSize, (int)1));
        LOGGER.info("Transpiled JavaScript file size:    " + Formatting.memoryFormat((long)jsSize, (int)1));
        LOGGER.info("JavaScript library file size:       " + Formatting.memoryFormat((long)libSize, (int)1));
        LOGGER.info("Results saved to " + this.outputDir.getAbsolutePath());
    }

    private void transpile() throws TeaVMToolException {
        LOGGER.info("Transpiling " + this.projectName + " to JavaScript");
        TeaVMTool transpiler = new TeaVMTool();
        transpiler.setClassLoader(this.getClass().getClassLoader());
        transpiler.setDebugInformationGenerated(true);
        transpiler.setIncremental(false);
        transpiler.setLog((TeaVMToolLog)new ConsoleTeaVMToolLog(true));
        transpiler.setMainClass(this.mainClassName);
        transpiler.setObfuscated(this.minify);
        transpiler.setSourceMapsFileGenerated(!this.minify);
        transpiler.setTargetDirectory(this.outputDir);
        transpiler.setTargetType(TeaVMTargetType.JAVASCRIPT);
        transpiler.generate();
        for (Problem problem : transpiler.getProblemProvider().getProblems()) {
            if (!this.shouldReport(problem)) continue;
            LOGGER.log(Level.WARNING, "Error while transpiling: " + this.format(problem));
        }
    }

    private boolean shouldReport(Problem problem) {
        String params = Arrays.toString(problem.getParams());
        return KNOWN_MISSING_CLASSES.stream().noneMatch(entry -> params.equals(entry));
    }

    private String format(Problem problem) {
        String text = problem.getText() + " " + Arrays.toString(problem.getParams());
        if (problem.getLocation() != null) {
            text = text + " (" + problem.getLocation().getSourceLocation() + ")";
        }
        return text;
    }

    protected void copyResources() {
        List<ResourceFile> resourceFiles = this.gatherResourceFiles();
        resourceFiles.add(this.generateManifest(resourceFiles));
        LOGGER.info("Copying " + resourceFiles.size() + " resource files");
        ArrayList<ResourceFile> textFiles = new ArrayList<ResourceFile>();
        for (ResourceFile file : resourceFiles) {
            if (this.isFileType(file, TEXT_FILE_TYPES)) {
                textFiles.add(file);
                continue;
            }
            this.copyBinaryResourceFile(file);
        }
        LOGGER.info("Generating " + FRAMEWORK_RESOURCE_FILE_PATHS.size() + " files");
        for (String path : FRAMEWORK_RESOURCE_FILE_PATHS) {
            if (path.endsWith(".html")) {
                this.rewriteHTML(new ResourceFile(path), textFiles);
                continue;
            }
            this.copyBinaryResourceFile(new ResourceFile(path));
        }
    }

    private boolean isFileType(ResourceFile needle, List<String> haystack) {
        return haystack.stream().anyMatch(type -> needle.getName().toLowerCase().endsWith((String)type));
    }

    private void rewriteHTML(ResourceFile file, List<ResourceFile> textResources) {
        File outputFile = this.getOutputFile(file);
        try (PrintWriter writer = new PrintWriter(outputFile, Charsets.UTF_8.displayName());){
            for (String line : file.readLines(Charsets.UTF_8)) {
                line = line.replace("{project}", this.projectName);
                if ((line = line.replace("{manifest}", this.generateManifestHTML())).trim().equals("{resources}")) {
                    line = this.generateTextResourceFilesHTML(textResources);
                }
                writer.println(line);
            }
        }
        catch (IOException e) {
            throw new MediaException("Cannot write to file: " + outputFile.getAbsolutePath(), e);
        }
    }

    private String generateManifestHTML() {
        if (this.manifestFile == null) {
            return "";
        }
        return "<link rel=\"manifest\" href=\"manifest.json\" />";
    }

    private String generateTextResourceFilesHTML(List<ResourceFile> files) {
        StringBuilder buffer = new StringBuilder();
        for (ResourceFile file : files) {
            Element element = new Element("div");
            element.setAttribute("id", this.normalizeFileName(file).replace(".", "_"));
            element.setText(file.read(Charsets.UTF_8));
            String xml = XMLHelper.toString((Element)element);
            buffer.append(xml);
            buffer.append("\n");
        }
        return buffer.toString();
    }

    private void copyBinaryResourceFile(ResourceFile file) {
        File outputFile = this.getOutputFile(file);
        try (InputStream stream = file.openStream();){
            byte[] contents = LoadUtils.readToByteArray((InputStream)stream);
            FileUtils.write((byte[])contents, (File)outputFile);
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, "Cannot copy file " + file, e);
        }
    }

    private void downloadJavaScriptLibrary(String url) {
        File librariesDir = new File(this.outputDir, "libraries");
        librariesDir.mkdir();
        String outputFileName = url.substring(url.lastIndexOf("/") + 1);
        File outputFile = new File(librariesDir, outputFileName);
        if (!outputFile.exists()) {
            LOGGER.info("Downloading " + url);
            try {
                URLResponse response = URLLoader.get((String)url).sendRequest();
                FileUtils.write((String)response.getBody(), (Charset)Charsets.UTF_8, (File)outputFile);
            }
            catch (IOException e) {
                throw new RuntimeException("Cannot download " + url, e);
            }
        }
    }

    private List<ResourceFile> gatherResourceFiles() {
        try {
            return Files.walk(this.resourceDir.toPath(), new FileVisitOption[0]).map(path -> path.toFile()).filter(file -> !file.isDirectory() && !file.getName().startsWith(".")).filter(file -> !file.getAbsolutePath().contains("/lib/")).map(file -> new ResourceFile(file)).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new MediaException("Cannot read resource files directory: " + this.resourceDir, e);
        }
    }

    private ResourceFile generateManifest(List<ResourceFile> resourceFiles) {
        try {
            File tempDir = Files.createTempDirectory("resource-file-manifest", new FileAttribute[0]).toFile();
            File manifestFile = new File(tempDir, "resource-file-manifest");
            List entries = resourceFiles.stream().map(file -> this.normalizeFileName((ResourceFile)file)).sorted().collect(Collectors.toList());
            Files.write(manifestFile.toPath(), entries, Charsets.UTF_8, new OpenOption[0]);
            return new ResourceFile(manifestFile);
        }
        catch (IOException e) {
            throw new MediaException("Cannot generate resource file manifest", e);
        }
    }

    private File getOutputFile(ResourceFile file) {
        if (this.isFileType(file, RESOURCE_FILE_TYPES)) {
            File resourcesDir = new File(this.outputDir, "resources");
            resourcesDir.mkdir();
            return new File(resourcesDir, this.normalizeFileName(file));
        }
        return new File(this.outputDir, this.normalizeFileName(file));
    }

    private String normalizeFileName(ResourceFile file) {
        return file.getName().replace("/", "_");
    }

    private void copyManifest() {
        try {
            File outputFile = new File(this.outputDir, "manifest.json");
            Files.copy(this.manifestFile.toPath(), outputFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            throw new MediaException("Cannot copy manifest", e);
        }
    }

    private void generateServiceWorker() {
        try {
            File outputFile = new File(this.outputDir, "service-worker.js");
            String template = Files.readString(outputFile.toPath(), Charsets.UTF_8);
            String resourceFilePaths = this.gatherServiceWorkerFilePaths().map(path -> "    \"" + path + "\",").collect(Collectors.joining("\n"));
            String generated = template.replace("{cachename}", this.projectName + "-" + System.currentTimeMillis()).replace("{resourcefiles}", resourceFilePaths);
            Files.write(outputFile.toPath(), generated.getBytes(Charsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new MediaException("Cannot create service worker", e);
        }
    }

    private Stream<String> gatherServiceWorkerFilePaths() throws IOException {
        return Files.walk(this.outputDir.toPath(), new FileVisitOption[0]).map(path -> this.outputDir.toPath().relativize((Path)path)).sorted().map(path -> "/" + path);
    }
}

