/*
 * Decompiled with CFR 0.152.
 */
package nl.colorize.multimedialib.scene;

import com.google.common.base.Preconditions;
import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import nl.colorize.multimedialib.math.MathUtils;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.DisplayMode;
import nl.colorize.multimedialib.renderer.FrameStats;
import nl.colorize.multimedialib.renderer.GraphicsMode;
import nl.colorize.multimedialib.renderer.InputDevice;
import nl.colorize.multimedialib.renderer.MediaLoader;
import nl.colorize.multimedialib.renderer.Network;
import nl.colorize.multimedialib.renderer.Renderer;
import nl.colorize.multimedialib.scene.Scene;
import nl.colorize.multimedialib.scene.Updatable;
import nl.colorize.multimedialib.stage.Stage;
import nl.colorize.util.LogHelper;
import nl.colorize.util.Platform;
import nl.colorize.util.Stopwatch;

public final class SceneContext
implements Updatable {
    private Renderer renderer;
    private Stopwatch timer;
    private long elapsedTime;
    private FrameStats frameStats;
    private Stage stage;
    private SceneLogic activeScene;
    private Queue<SceneLogic> requestedSceneQueue;
    private List<Scene> globalScenes;
    private int lastCanvasWidth;
    private int lastCanvasHeight;
    private static final long FRAME_LEEWAY_MS = 5L;
    private static final float MIN_FRAME_TIME = 0.01f;
    private static final float MAX_FRAME_TIME = 0.2f;
    private static final int RESIZE_TOLERANCE = 20;
    private static final Logger LOGGER = LogHelper.getLogger(SceneContext.class);

    public SceneContext(Renderer renderer, Stopwatch timer) {
        this.renderer = renderer;
        this.timer = timer;
        this.elapsedTime = 0L;
        this.frameStats = new FrameStats(renderer.getDisplayMode());
        this.stage = new Stage(renderer.getGraphicsMode(), renderer.getCanvas());
        this.requestedSceneQueue = new LinkedList<SceneLogic>();
        this.globalScenes = new ArrayList<Scene>();
        this.lastCanvasWidth = renderer.getCanvas().getWidth();
        this.lastCanvasHeight = renderer.getCanvas().getHeight();
    }

    public boolean syncFrame() {
        long frameTime = this.timer.tick();
        this.elapsedTime += frameTime;
        if (this.elapsedTime < (long)this.getDisplayMode().getFrameTimeMS() - 5L) {
            return false;
        }
        this.frameStats.markEnd("$$frameTime");
        float deltaTime = MathUtils.clamp((float)this.elapsedTime / 1000.0f, 0.01f, 0.2f);
        this.frameStats.markStart("$$frameUpdate");
        this.update(deltaTime);
        this.frameStats.markEnd("$$frameUpdate");
        this.elapsedTime = 0L;
        return true;
    }

    @Override
    public void update(float deltaTime) {
        this.getInput().update(deltaTime);
        this.stage.update(deltaTime);
        if (!this.requestedSceneQueue.isEmpty()) {
            this.activateRequestedScene();
        }
        this.updateSceneGraph(this.activeScene, deltaTime);
        this.updateGlobalScenes(deltaTime);
        this.lastCanvasWidth = this.getCanvas().getWidth();
        this.lastCanvasHeight = this.getCanvas().getHeight();
    }

    private void updateSceneGraph(SceneLogic current, float deltaTime) {
        this.checkCanvasResize(current);
        current.scene.update(this, deltaTime);
        for (int i = current.subScenes.size() - 1; i >= 0; --i) {
            Scene subScene = current.subScenes.get(i);
            if (this.checkCompleted(current, subScene)) continue;
            subScene.update(this, deltaTime);
            this.checkCompleted(current, subScene);
        }
    }

    private void checkCanvasResize(SceneLogic current) {
        int width = this.getCanvas().getWidth();
        int height = this.getCanvas().getHeight();
        if (Math.abs(width - this.lastCanvasWidth) >= 20 || Math.abs(height - this.lastCanvasHeight) >= 20) {
            current.scene.resize(this, width, height);
        }
    }

    private boolean checkCompleted(SceneLogic parent, Scene subScene) {
        if (subScene.isCompleted()) {
            subScene.end(this);
            parent.subScenes.remove(subScene);
            return true;
        }
        return false;
    }

    private void activateRequestedScene() {
        SceneLogic requestedScene;
        if (this.activeScene != null) {
            this.activeScene.walk(scene -> scene.end(this));
            this.stage.clear();
        }
        if ((requestedScene = this.requestedSceneQueue.peek()) != null) {
            this.activeScene = requestedScene;
            this.activeScene.walk(scene -> scene.start(this));
            this.requestedSceneQueue.poll();
            if (!this.requestedSceneQueue.isEmpty()) {
                this.activateRequestedScene();
            }
        }
    }

    private void updateGlobalScenes(float deltaTime) {
        Iterator<Scene> iterator = this.globalScenes.iterator();
        while (iterator.hasNext()) {
            Scene globalScene = iterator.next();
            globalScene.update(this, deltaTime);
            if (!globalScene.isCompleted()) continue;
            iterator.remove();
        }
    }

    public void changeScene(Scene requestedScene) {
        this.requestedSceneQueue.offer(new SceneLogic(requestedScene));
    }

    public void attach(Scene subScene) {
        if (this.requestedSceneQueue.isEmpty()) {
            this.activeScene.attachSubScene(subScene);
            subScene.start(this);
        } else {
            SceneLogic requestedScene = this.requestedSceneQueue.peek();
            requestedScene.attachSubScene(subScene);
        }
    }

    public void attach(Updatable subScene) {
        Scene wrappedSubScene = (context, deltaTime) -> subScene.update(deltaTime);
        this.attach(wrappedSubScene);
    }

    public void attach(Runnable subScene) {
        Scene wrappedSubScene = (context, deltaTime) -> subScene.run();
        this.attach(wrappedSubScene);
    }

    public void attachGlobal(Scene globalScene) {
        this.globalScenes.add(globalScene);
        globalScene.start(this);
    }

    public String getRendererName() {
        if (this.renderer.getGraphics() == null) {
            return "<headless>";
        }
        return this.renderer.getGraphics().getClass().getSimpleName().replace("GraphicsContext", "").replace("Graphics", "");
    }

    public List<String> getDebugInformation() {
        int targetFPS = this.frameStats.getTargetFramerate();
        ArrayList<String> info = new ArrayList<String>();
        info.add("Renderer:  " + this.getRendererName());
        info.add("Canvas:  " + this.getCanvas());
        info.add("Framerate:  " + this.frameStats.getActualFramerate() + " / " + targetFPS);
        info.add("Update time:  " + this.frameStats.getFrameUpdateTime() + "ms");
        info.add("Render time:  " + this.frameStats.getFrameRenderTime() + "ms");
        if (!this.frameStats.getCustomStats().isEmpty()) {
            info.add("--------");
        }
        for (String customStat : this.frameStats.getCustomStats()) {
            info.add(customStat + ":  " + this.frameStats.getAverageTimeMS(customStat) + "ms");
        }
        return info;
    }

    public void terminate() {
        this.renderer.terminate();
    }

    public GraphicsMode getGraphicsMode() {
        return this.renderer.getGraphicsMode();
    }

    public DisplayMode getDisplayMode() {
        return this.renderer.getDisplayMode();
    }

    public Canvas getCanvas() {
        return this.renderer.getCanvas();
    }

    public InputDevice getInput() {
        return this.renderer.getInput();
    }

    public MediaLoader getMediaLoader() {
        return this.renderer.getMediaLoader();
    }

    public Network getNetwork() {
        return this.renderer.getNetwork();
    }

    public void takeScreenshot() {
        Preconditions.checkState((Platform.isWindows() || Platform.isMac() ? 1 : 0) != 0, (Object)("Taking screenshots is not supported on platform " + Platform.getPlatform()));
        try {
            File outputFile = new File(Platform.getUserDesktopDir(), "screenshot-" + System.currentTimeMillis() + ".png");
            this.renderer.takeScreenshot(outputFile);
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Failed to save screenshot", e);
        }
    }

    public FrameStats getFrameStats() {
        return this.frameStats;
    }

    public Stage getStage() {
        return this.stage;
    }

    private static class SceneLogic {
        private Scene scene;
        private List<Scene> subScenes;

        public SceneLogic(Scene scene) {
            this.scene = scene;
            this.subScenes = new ArrayList<Scene>();
        }

        public void attachSubScene(Scene subScene) {
            this.subScenes.add(0, subScene);
        }

        public void walk(Consumer<Scene> callback) {
            callback.accept(this.scene);
            for (int i = this.subScenes.size() - 1; i >= 0; --i) {
                callback.accept(this.subScenes.get(i));
            }
        }
    }
}

