/*
 * Decompiled with CFR 0.152.
 */
package nl.colorize.multimedialib.scene;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.function.Consumer;
import java.util.logging.Logger;
import nl.colorize.multimedialib.renderer.Canvas;
import nl.colorize.multimedialib.renderer.DisplayMode;
import nl.colorize.multimedialib.renderer.FrameStats;
import nl.colorize.multimedialib.renderer.GraphicsMode;
import nl.colorize.multimedialib.renderer.InputDevice;
import nl.colorize.multimedialib.renderer.MediaLoader;
import nl.colorize.multimedialib.renderer.Network;
import nl.colorize.multimedialib.renderer.Renderer;
import nl.colorize.multimedialib.scene.Scene;
import nl.colorize.multimedialib.scene.Timer;
import nl.colorize.multimedialib.scene.Updatable;
import nl.colorize.multimedialib.stage.Stage;
import nl.colorize.util.LogHelper;
import nl.colorize.util.Stopwatch;

public final class SceneContext
implements Updatable {
    private Renderer renderer;
    private GraphicsMode graphicsMode;
    private DisplayMode displayMode;
    private Canvas canvas;
    private Stage stage;
    private MediaLoader mediaLoader;
    private InputDevice input;
    private Network network;
    private Stopwatch timer;
    private long elapsedTime;
    private FrameStats frameStats;
    private SceneState activeScene;
    private Queue<SceneState> requestedSceneQueue;
    private List<Scene> globalSubScenes;
    private int lastCanvasWidth;
    private int lastCanvasHeight;
    private static final long FRAME_LEEWAY_MS = 5L;
    private static final float MIN_FRAME_TIME = 0.01f;
    private static final float MAX_FRAME_TIME = 0.2f;
    private static final int RESIZE_TOLERANCE = 20;
    private static final Logger LOGGER = LogHelper.getLogger(SceneContext.class);

    public SceneContext(Renderer renderer, MediaLoader media, InputDevice input, Network network) {
        Preconditions.checkArgument((renderer != null ? 1 : 0) != 0, (Object)"Context not attached to renderer");
        this.renderer = renderer;
        this.graphicsMode = renderer.getGraphicsMode();
        this.displayMode = renderer.getDisplayMode();
        this.canvas = this.displayMode.canvas();
        this.mediaLoader = media;
        this.input = input;
        this.network = network;
        this.timer = new Stopwatch();
        this.elapsedTime = 0L;
        this.frameStats = new FrameStats(this.displayMode);
        this.requestedSceneQueue = new LinkedList<SceneState>();
        this.globalSubScenes = new ArrayList<Scene>();
        this.lastCanvasWidth = this.canvas.getWidth();
        this.lastCanvasHeight = this.canvas.getHeight();
        this.stage = new Stage(this.graphicsMode, this.canvas, this.frameStats);
    }

    @VisibleForTesting
    protected void replaceTimer(Stopwatch timer) {
        this.timer = timer;
    }

    public int syncFrame() {
        long frameTime = this.timer.tick();
        this.elapsedTime += frameTime;
        if (this.elapsedTime < (long)this.getDisplayMode().getFrameTimeMS() - 5L) {
            return 0;
        }
        this.frameStats.markEnd("$$frameTime");
        this.frameStats.resetGraphicsCount();
        float deltaTime = Math.clamp((float)this.elapsedTime / 1000.0f, 0.01f, 0.2f);
        this.frameStats.markStart("$$frameUpdate");
        this.update(deltaTime);
        this.frameStats.markEnd("$$frameUpdate");
        this.elapsedTime = 0L;
        return 1;
    }

    @Override
    public void update(float deltaTime) {
        this.getInput().update(deltaTime);
        if (!this.requestedSceneQueue.isEmpty()) {
            this.activateRequestedScene();
        }
        this.updateCurrentScene(this.activeScene, deltaTime);
        this.updateGlobalSubScenes(deltaTime);
        this.lastCanvasWidth = this.getCanvas().getWidth();
        this.lastCanvasHeight = this.getCanvas().getHeight();
    }

    private void updateCurrentScene(SceneState current, float deltaTime) {
        this.checkCanvasResize(current);
        current.sceneTimer.update(deltaTime);
        current.scene.update(this, deltaTime);
        for (int i = current.subScenes.size() - 1; i >= 0; --i) {
            Scene subScene = current.subScenes.get(i);
            if (this.checkCompleted(current, subScene)) continue;
            subScene.update(this, deltaTime);
            this.checkCompleted(current, subScene);
        }
    }

    private void checkCanvasResize(SceneState current) {
        int width = this.getCanvas().getWidth();
        int height = this.getCanvas().getHeight();
        if (Math.abs(width - this.lastCanvasWidth) >= 20 || Math.abs(height - this.lastCanvasHeight) >= 20) {
            current.scene.resize(this, width, height);
        }
    }

    private boolean checkCompleted(SceneState parent, Scene subScene) {
        if (subScene.isCompleted()) {
            subScene.end(this);
            parent.subScenes.remove(subScene);
            return true;
        }
        return false;
    }

    private void activateRequestedScene() {
        SceneState requestedScene;
        if (this.activeScene != null) {
            this.activeScene.walk(scene -> scene.end(this));
            this.stage.clear();
        }
        if ((requestedScene = this.requestedSceneQueue.peek()) != null) {
            this.activeScene = requestedScene;
            this.activeScene.walk(scene -> scene.start(this));
            this.requestedSceneQueue.poll();
            if (!this.requestedSceneQueue.isEmpty()) {
                this.activateRequestedScene();
            }
        }
    }

    private void updateGlobalSubScenes(float deltaTime) {
        Iterator<Scene> iterator = this.globalSubScenes.iterator();
        while (iterator.hasNext()) {
            Scene globalScene = iterator.next();
            globalScene.update(this, deltaTime);
            if (!globalScene.isCompleted()) continue;
            iterator.remove();
        }
    }

    public void changeScene(Scene requestedScene) {
        this.requestedSceneQueue.offer(new SceneState(requestedScene));
    }

    public void attach(Scene subScene) {
        if (this.requestedSceneQueue.isEmpty()) {
            this.activeScene.attachSubScene(subScene);
            subScene.start(this);
        } else {
            SceneState requestedScene = this.requestedSceneQueue.peek();
            requestedScene.attachSubScene(subScene);
        }
    }

    public void attach(Updatable subScene) {
        Scene wrappedSubScene = (context, deltaTime) -> subScene.update(deltaTime);
        this.attach(wrappedSubScene);
    }

    public void attach(Runnable subScene) {
        Scene wrappedSubScene = (context, deltaTime) -> subScene.run();
        this.attach(wrappedSubScene);
    }

    public void attachGlobal(Scene globalSubScene) {
        this.globalSubScenes.add(globalSubScene);
        globalSubScene.start(this);
    }

    public Timer getSceneTime() {
        if (this.activeScene == null) {
            LOGGER.warning("Scene timer is not yet available");
            return Timer.none();
        }
        return this.activeScene.sceneTimer;
    }

    public String getRendererName() {
        return this.renderer.toString().replace("Renderer", "").replace("renderer", "").trim();
    }

    public List<String> getDebugInformation() {
        int targetFPS = this.frameStats.getTargetFramerate();
        ArrayList<String> info = new ArrayList<String>();
        info.add("Renderer:  " + this.getRendererName());
        info.add("Canvas:  " + String.valueOf(this.getCanvas()));
        info.add("Framerate:  " + Math.round(this.frameStats.getAverageFramerate()) + " / " + targetFPS);
        info.add("Update time:  " + this.frameStats.getFrameUpdateTime() + "ms");
        info.add("Render time:  " + this.frameStats.getFrameRenderTime() + "ms");
        if (!this.frameStats.getCustomStats().isEmpty()) {
            info.add("");
        }
        for (String customStat : this.frameStats.getCustomStats()) {
            info.add(customStat + ":  " + this.frameStats.getAverageTimeMS(customStat) + "ms");
        }
        return info;
    }

    public void terminate() {
        this.renderer.terminate();
    }

    public GraphicsMode getGraphicsMode() {
        return this.graphicsMode;
    }

    public DisplayMode getDisplayMode() {
        return this.displayMode;
    }

    public Canvas getCanvas() {
        return this.canvas;
    }

    public Stage getStage() {
        return this.stage;
    }

    public MediaLoader getMediaLoader() {
        return this.mediaLoader;
    }

    public InputDevice getInput() {
        return this.input;
    }

    public Network getNetwork() {
        return this.network;
    }

    public FrameStats getFrameStats() {
        return this.frameStats;
    }

    private static class SceneState {
        private Scene scene;
        private List<Scene> subScenes;
        private Timer sceneTimer;

        public SceneState(Scene scene) {
            this.scene = scene;
            this.subScenes = new ArrayList<Scene>();
            this.sceneTimer = Timer.infinite();
        }

        public void attachSubScene(Scene subScene) {
            this.subScenes.addFirst(subScene);
        }

        public void walk(Consumer<Scene> callback) {
            callback.accept(this.scene);
            for (int i = this.subScenes.size() - 1; i >= 0; --i) {
                callback.accept(this.subScenes.get(i));
            }
        }
    }
}

