/*
 * Decompiled with CFR 0.152.
 */
package nl.colorize.multimedialib.tool;

import com.google.common.base.CharMatcher;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.xml.XmlEscapers;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import nl.colorize.multimedialib.renderer.MediaException;
import nl.colorize.util.DateParser;
import nl.colorize.util.FileUtils;
import nl.colorize.util.LogHelper;
import nl.colorize.util.ResourceFile;
import nl.colorize.util.Stopwatch;
import nl.colorize.util.cli.Arg;
import nl.colorize.util.cli.CommandLineArgumentParser;
import nl.colorize.util.cli.CommandLineInterfaceException;
import nl.colorize.util.http.URLLoader;
import nl.colorize.util.http.URLResponse;
import org.teavm.diagnostics.Problem;
import org.teavm.tooling.ConsoleTeaVMToolLog;
import org.teavm.tooling.TeaVMTargetType;
import org.teavm.tooling.TeaVMTool;
import org.teavm.tooling.TeaVMToolException;
import org.teavm.tooling.TeaVMToolLog;

public class TeaVMTranspilerTool {
    @Arg(name="project", usage="Project display name")
    protected String projectName;
    @Arg(name="main", usage="Main class that acts as application entry point")
    protected String mainClassName;
    @Arg(name="resources", usage="Location of the application's resource files")
    protected File resourceDir;
    @Arg(name="out", usage="Output directory for transpiled application")
    protected File outputDir;
    @Arg(usage="Minifies the generated JavaScript, off by default")
    protected boolean minify;
    @Arg(usage="Inserts <meta> tags into the HTML, passed as name=value.", required=false)
    protected String meta;
    private static final ResourceFile INDEX_FILE = new ResourceFile("browser/index.html");
    private static final ResourceFile RESOURCES_LIST = new ResourceFile("browser/browser-resources.txt");
    private static final ResourceFile JS_LIST = new ResourceFile("browser/javascript-libraries.txt");
    private static final String SCRIPT_FILE_NAME = "script-" + System.currentTimeMillis() + ".js";
    private static final List<String> EXPECTED_RESOURCES = List.of("favicon.png", "apple-favicon.png");
    private static final Logger LOGGER = LogHelper.getLogger(TeaVMTranspilerTool.class);
    private static final List<String> TEXT_FILE_TYPES = List.of(".atlas", ".csv", ".fnt", ".glsl", ".json", ".md", ".properties", ".txt", ".yaml", ".yml", "-manifest");
    private static final List<String> RESOURCE_FILE_TYPES = List.of(".css", ".fbx", ".g3db", ".gif", ".gltf", ".jpg", ".mp3", ".ogg", ".png", ".svg", ".ttf", ".wav");
    private static final List<String> KNOWN_MISSING_CLASSES = List.of("[java.lang.System.exit(I)V]", "[java.lang.reflect.TypeVariable]", "[java.nio.file.Path]", "[java.io.File.toPath()Ljava/nio/file/Path;]", "[java.lang.Class.getGenericSuperclass()Ljava/lang/reflect/Type;]", "[java.util.Properties.load(Ljava/io/Reader;)V]", "[java.util.TimeZone.toZoneId()Ljava/time/ZoneId;]", "[java.util.concurrent.CopyOnWriteArrayList]");

    public static void main(String[] argv) {
        CommandLineArgumentParser argParser = new CommandLineArgumentParser("TeaVMTranspilerTool");
        TeaVMTranspilerTool transpiler = (TeaVMTranspilerTool)argParser.parse(argv, TeaVMTranspilerTool.class);
        transpiler.run();
    }

    protected void run() {
        Preconditions.checkArgument((boolean)this.resourceDir.exists(), (Object)("Resource directory not found: " + this.resourceDir.getAbsolutePath()));
        this.outputDir.mkdir();
        this.checkMainClass();
        try {
            this.cleanOldScripts();
            this.copyResources();
            this.transpile();
            this.printSummary();
        }
        catch (IOException | TeaVMToolException e) {
            LOGGER.log(Level.SEVERE, "Transpiling failed", e);
        }
    }

    private void checkMainClass() {
        try {
            Class<?> mainClass = Class.forName(this.mainClassName);
            mainClass.getDeclaredMethod("main", String[].class);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Invalid main class: " + this.mainClassName, e);
        }
    }

    private void printSummary() throws IOException {
        long htmlSize = new File(this.outputDir, "index.html").length();
        long jsSize = this.getScriptFile().length();
        long resourceSize = FileUtils.countDirectorySize((File)new File(this.outputDir, "resources"));
        LOGGER.info("HTML file size:                   " + FileUtils.formatFileSize((long)htmlSize));
        LOGGER.info("Transpiled JavaScript file size:  " + FileUtils.formatFileSize((long)jsSize));
        LOGGER.info("Resource file size:               " + FileUtils.formatFileSize((long)resourceSize));
        LOGGER.info("Results saved to " + this.outputDir.getAbsolutePath());
    }

    private void transpile() throws TeaVMToolException {
        Stopwatch timer = new Stopwatch();
        LOGGER.info("Transpiling " + this.projectName + " to JavaScript");
        TeaVMTool transpiler = new TeaVMTool();
        transpiler.setClassLoader(this.getClass().getClassLoader());
        transpiler.setDebugInformationGenerated(true);
        transpiler.setIncremental(false);
        transpiler.setLog((TeaVMToolLog)new ConsoleTeaVMToolLog(true));
        transpiler.setMainClass(this.mainClassName);
        transpiler.setObfuscated(this.minify);
        transpiler.setSourceMapsFileGenerated(!this.minify);
        transpiler.setTargetDirectory(this.outputDir);
        transpiler.setTargetType(TeaVMTargetType.JAVASCRIPT);
        transpiler.setTargetFileName(SCRIPT_FILE_NAME);
        transpiler.generate();
        this.checkTranspilerOutput(transpiler);
        LOGGER.info("Transpilation took " + Math.round((float)timer.tock() / 1000.0f) + "s");
    }

    private void checkTranspilerOutput(TeaVMTool transpiler) {
        for (Problem problem : transpiler.getProblemProvider().getProblems()) {
            if (!this.shouldReport(problem)) continue;
            String details = this.format(problem);
            throw new UnsupportedOperationException("TeaVM transpiler error:\n" + details);
        }
    }

    private boolean shouldReport(Problem problem) {
        String params = Arrays.toString(problem.getParams());
        return KNOWN_MISSING_CLASSES.stream().noneMatch(entry -> params.equals(entry));
    }

    private String format(Problem problem) {
        String text = problem.getText() + " " + Arrays.toString(problem.getParams());
        if (problem.getLocation() != null) {
            text = text + " (" + String.valueOf(problem.getLocation().getSourceLocation()) + ")";
        }
        return text;
    }

    protected void copyResources() {
        List<ResourceFile> applicationResourceFiles = this.gatherApplicationResourceFiles();
        this.inspectApplicationResourceFiles(applicationResourceFiles);
        ArrayList<ResourceFile> resourceFiles = new ArrayList<ResourceFile>();
        resourceFiles.addAll(this.gatherFrameworkResourceFiles());
        resourceFiles.addAll(applicationResourceFiles);
        resourceFiles.add(this.generateManifest(resourceFiles));
        LOGGER.info("Copying " + resourceFiles.size() + " resource files");
        ArrayList<ResourceFile> textFiles = new ArrayList<ResourceFile>();
        List<String> jsLibraries = this.copyJavaScriptLibraries();
        for (ResourceFile file : resourceFiles) {
            if (this.isFileType(file, TEXT_FILE_TYPES)) {
                textFiles.add(file);
                continue;
            }
            this.copyBinaryResourceFile(file);
        }
        this.rewriteHTML(textFiles, jsLibraries);
    }

    private List<String> copyJavaScriptLibraries() {
        return JS_LIST.readLines(Charsets.UTF_8).stream().filter(line -> !line.isEmpty() && !line.startsWith("#")).map(line -> this.copyJavaScriptLibrary((String)line).getName()).toList();
    }

    private File copyJavaScriptLibrary(String url) {
        File libDir = new File(this.outputDir, "libraries");
        libDir.mkdir();
        try {
            URLLoader request = URLLoader.get((String)url);
            URLResponse response = request.send();
            File outputFile = new File(libDir, url.substring(url.lastIndexOf("/") + 1));
            FileUtils.write((byte[])response.getBody(), (File)outputFile);
            return outputFile;
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to download " + url, e);
        }
    }

    private boolean isFileType(ResourceFile needle, List<String> haystack) {
        return haystack.stream().anyMatch(type -> needle.getName().toLowerCase().endsWith((String)type));
    }

    private void rewriteHTML(List<ResourceFile> textFiles, List<String> jsLibraries) {
        File outputFile = this.getOutputFile(INDEX_FILE);
        try (PrintWriter writer = new PrintWriter(outputFile, Charsets.UTF_8.displayName());){
            for (String line : INDEX_FILE.readLines(Charsets.UTF_8)) {
                line = line.replace("{project}", this.projectName);
                line = line.replace("{js-libraries}", this.generateScriptTags(jsLibraries));
                line = line.replace("{teavm-js-file}", SCRIPT_FILE_NAME);
                line = line.replace("{timestamp}", this.generateTimestampTag());
                if ((line = line.replace("{meta}", this.generateMetaTags())).trim().equals("{resources}")) {
                    line = this.generateTextResourceFilesHTML(textFiles);
                }
                writer.println(line);
            }
        }
        catch (IOException e) {
            throw new MediaException("Cannot write to file: " + outputFile.getAbsolutePath(), e);
        }
    }

    private String generateScriptTags(List<String> jsLibraries) {
        return jsLibraries.stream().map(file -> "<script src=\"libraries/" + file + "\"></script>").collect(Collectors.joining("\n"));
    }

    private String generateTimestampTag() {
        String timestamp = DateParser.format((Date)new Date(), (String)"yyyy-MM-dd HH:mm");
        return "<!-- Generated by Colorize MultimediaLib @ " + timestamp + " -->";
    }

    private String generateMetaTags() {
        if (this.meta == null || this.meta.isEmpty()) {
            return "";
        }
        return Splitter.on((String)",").splitToStream((CharSequence)this.meta).map(this::generateMetaTag).collect(Collectors.joining("\n"));
    }

    private String generateMetaTag(String entry) {
        List parts = Splitter.on((CharMatcher)CharMatcher.anyOf((CharSequence)"=:")).splitToList((CharSequence)entry);
        if (parts.size() != 2) {
            throw new CommandLineInterfaceException("Invalid meta tag: " + entry);
        }
        return String.format("<meta name=\"%s\" content=\"%s\" />", parts.get(0), parts.get(1));
    }

    private String generateTextResourceFilesHTML(List<ResourceFile> files) {
        StringBuilder buffer = new StringBuilder();
        for (ResourceFile file : files) {
            String id = this.normalizeFileName(file).replace(".", "_");
            String contents = file.read(Charsets.UTF_8);
            buffer.append("<div id=\"" + id + "\">");
            buffer.append(XmlEscapers.xmlContentEscaper().escape(contents));
            buffer.append("</div>\n");
        }
        return buffer.toString();
    }

    private void copyBinaryResourceFile(ResourceFile file, File outputFile) {
        try (InputStream stream = file.openStream();){
            byte[] contents = stream.readAllBytes();
            FileUtils.write((byte[])contents, (File)outputFile);
        }
        catch (IOException e) {
            throw new RuntimeException("Cannot copy file: " + String.valueOf(file), e);
        }
    }

    private void copyBinaryResourceFile(ResourceFile file) {
        File outputFile = this.getOutputFile(file);
        this.copyBinaryResourceFile(file, outputFile);
    }

    private List<ResourceFile> gatherFrameworkResourceFiles() {
        return RESOURCES_LIST.readLines(Charsets.UTF_8).stream().filter(line -> !line.isEmpty() && !line.startsWith("#")).map(ResourceFile::new).toList();
    }

    private List<ResourceFile> gatherApplicationResourceFiles() {
        try {
            return Files.walk(this.resourceDir.toPath(), new FileVisitOption[0]).map(path -> path.toFile()).filter(file -> !file.isDirectory() && !file.getName().startsWith(".")).filter(file -> !file.getAbsolutePath().contains("/lib/")).map(ResourceFile::new).toList();
        }
        catch (IOException e) {
            throw new MediaException("Cannot read resource files directory: " + String.valueOf(this.resourceDir), e);
        }
    }

    private ResourceFile generateManifest(List<ResourceFile> resourceFiles) {
        try {
            File tempDir = Files.createTempDirectory("resource-file-manifest", new FileAttribute[0]).toFile();
            File manifestFile = new File(tempDir, "resource-file-manifest");
            List<String> entries = resourceFiles.stream().map(file -> this.normalizeFileName((ResourceFile)file)).filter(file -> !file.endsWith(".js")).distinct().sorted().toList();
            Files.write(manifestFile.toPath(), entries, Charsets.UTF_8, new OpenOption[0]);
            return new ResourceFile(manifestFile);
        }
        catch (IOException e) {
            throw new MediaException("Cannot generate resource file manifest", e);
        }
    }

    private File getOutputFile(ResourceFile file) {
        if (this.isFileType(file, RESOURCE_FILE_TYPES)) {
            File resourcesDir = new File(this.outputDir, "resources");
            resourcesDir.mkdir();
            return new File(resourcesDir, this.normalizeFileName(file));
        }
        return new File(this.outputDir, this.normalizeFileName(file));
    }

    protected File getScriptFile() {
        return new File(this.outputDir, SCRIPT_FILE_NAME);
    }

    private String normalizeFileName(ResourceFile file) {
        return file.getName().replace("/", "_");
    }

    private void cleanOldScripts() throws IOException {
        for (File file : FileUtils.walkFiles((File)this.outputDir, f -> f.getName().startsWith("script-"))) {
            FileUtils.delete((File)file);
        }
    }

    private void inspectApplicationResourceFiles(List<ResourceFile> resourceFiles) {
        List<String> fileNames = resourceFiles.stream().map(ResourceFile::getName).toList();
        for (String expected : EXPECTED_RESOURCES) {
            if (fileNames.contains(expected)) continue;
            LOGGER.warning("Missing resource file " + expected);
        }
    }
}

