/*
 * Decompiled with CFR 0.152.
 */
package nl.colorize.multimedialib.tool;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.xml.XmlEscapers;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import nl.colorize.multimedialib.renderer.MediaException;
import nl.colorize.multimedialib.tool.TeaDemoLauncher;
import nl.colorize.util.DateParser;
import nl.colorize.util.FileUtils;
import nl.colorize.util.LogHelper;
import nl.colorize.util.ResourceFile;
import nl.colorize.util.Stopwatch;
import nl.colorize.util.cli.Arg;
import nl.colorize.util.cli.CommandLineArgumentParser;
import nl.colorize.util.cli.CommandLineInterfaceException;
import nl.colorize.util.http.URLLoader;
import org.teavm.diagnostics.DefaultProblemTextConsumer;
import org.teavm.diagnostics.Problem;
import org.teavm.diagnostics.ProblemTextConsumer;
import org.teavm.tooling.ConsoleTeaVMToolLog;
import org.teavm.tooling.TeaVMTargetType;
import org.teavm.tooling.TeaVMTool;
import org.teavm.tooling.TeaVMToolException;
import org.teavm.tooling.TeaVMToolLog;

public class TeaVMTranspilerTool {
    @Arg(name="project", usage="Project display name")
    protected String projectName;
    @Arg(name="main", usage="Main class that acts as application entry point")
    protected String mainClassName;
    @Arg(name="resources", usage="Location of the application's resource files")
    protected File resourceDir;
    @Arg(name="out", usage="Output directory for transpiled application")
    protected File outputDir;
    @Arg(usage="Build ID used for caching resource files, default is random")
    protected String buildId = String.valueOf(System.currentTimeMillis());
    @Arg(usage="Minifies the generated JavaScript, off by default")
    protected boolean minify;
    @Arg(usage="Inserts <meta> tags into the HTML, passed as name=value.", required=false)
    protected String meta;
    @Arg(usage="Overrides the application with the demo application, for testing purposes.")
    protected boolean demo;
    private static final ResourceFile INDEX_FILE = new ResourceFile("browser/index.html");
    private static final ResourceFile RESOURCES_LIST = new ResourceFile("browser/browser-resources.txt");
    private static final ResourceFile JS_LIST = new ResourceFile("browser/javascript-libraries.txt");
    private static final Splitter JS_LIB_SPLITTER = Splitter.on((String)"->").trimResults();
    private static final String SCRIPT_FILE_NAME = "script-" + System.currentTimeMillis() + ".js";
    private static final List<String> EXPECTED_RESOURCES = List.of("favicon.png", "apple-favicon.png");
    private static final Splitter PATH_SPLITTER = Splitter.on((String)"/").omitEmptyStrings();
    private static final Logger LOGGER = LogHelper.getLogger(TeaVMTranspilerTool.class);
    private static final List<String> TEXT_FILE_TYPES = List.of(".atlas", ".csv", ".fnt", ".glsl", ".json", ".md", ".properties", ".txt", ".yaml", ".yml", "-manifest");
    private static final List<String> RESOURCE_FILE_TYPES = List.of(".css", ".fbx", ".g3db", ".gif", ".gltf", ".jpg", ".mp3", ".mtl", ".obj", ".ogg", ".png", ".svg", ".ttf", ".wav");

    public static void main(String[] argv) {
        CommandLineArgumentParser argParser = new CommandLineArgumentParser("TeaVMTranspilerTool");
        TeaVMTranspilerTool transpiler = (TeaVMTranspilerTool)argParser.parse(argv, TeaVMTranspilerTool.class);
        transpiler.run();
    }

    protected void run() {
        Preconditions.checkArgument((boolean)this.resourceDir.exists(), (Object)("Resource directory not found: " + this.resourceDir.getAbsolutePath()));
        this.outputDir.mkdir();
        this.checkMainClass();
        try {
            this.cleanOldScripts();
            this.copyResources();
            this.transpile();
            this.printSummary();
        }
        catch (IOException | TeaVMToolException e) {
            LOGGER.log(Level.SEVERE, "Transpiling failed", e);
        }
    }

    private void checkMainClass() {
        if (this.demo) {
            this.mainClassName = TeaDemoLauncher.class.getName();
        }
        try {
            Class<?> mainClass = Class.forName(this.mainClassName);
            mainClass.getDeclaredMethod("main", String[].class);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Invalid main class: " + this.mainClassName, e);
        }
    }

    private void printSummary() throws IOException {
        long htmlSize = new File(this.outputDir, "index.html").length();
        long jsSize = this.getScriptFile().length();
        long resourceSize = FileUtils.countDirectorySize((File)new File(this.outputDir, "resources"));
        LOGGER.info("HTML file size:                   " + FileUtils.formatFileSize((long)htmlSize));
        LOGGER.info("Transpiled JavaScript file size:  " + FileUtils.formatFileSize((long)jsSize));
        LOGGER.info("Resource file size:               " + FileUtils.formatFileSize((long)resourceSize));
        LOGGER.info("Results saved to " + this.outputDir.getAbsolutePath());
    }

    private void transpile() throws TeaVMToolException {
        Stopwatch timer = new Stopwatch();
        LOGGER.info("Transpiling " + this.projectName + " to JavaScript");
        TeaVMTool transpiler = new TeaVMTool();
        transpiler.setClassLoader(this.getClass().getClassLoader());
        transpiler.setIncremental(false);
        transpiler.setLog((TeaVMToolLog)new ConsoleTeaVMToolLog(true));
        transpiler.setMainClass(this.mainClassName);
        transpiler.setEntryPointName("main");
        transpiler.setObfuscated(this.minify);
        transpiler.setSourceMapsFileGenerated(!this.minify);
        transpiler.setTargetDirectory(this.outputDir);
        transpiler.setTargetType(TeaVMTargetType.JAVASCRIPT);
        transpiler.setTargetFileName(SCRIPT_FILE_NAME);
        transpiler.generate();
        this.checkTranspilerOutput(transpiler);
        LOGGER.info("Transpilation took " + Math.round((float)timer.tock() / 1000.0f) + "s");
    }

    private void checkTranspilerOutput(TeaVMTool transpiler) {
        String details;
        for (Problem problem : transpiler.getProblemProvider().getProblems()) {
            details = this.format(problem);
            LOGGER.warning("TeaVM transpiler warning:\n" + details);
        }
        Iterator iterator = transpiler.getProblemProvider().getSevereProblems().iterator();
        if (iterator.hasNext()) {
            Problem problem;
            problem = (Problem)iterator.next();
            details = this.format(problem);
            throw new UnsupportedOperationException("TeaVM transpiler error:\n" + details);
        }
    }

    private String format(Problem problem) {
        DefaultProblemTextConsumer textRenderer = new DefaultProblemTextConsumer();
        problem.render((ProblemTextConsumer)textRenderer);
        String text = "    " + textRenderer.getText();
        if (problem.getLocation() != null) {
            text = text + "\n    (" + String.valueOf(problem.getLocation().getSourceLocation()) + ")";
        }
        return text;
    }

    protected void copyResources() {
        List<ResourceFile> applicationResourceFiles = this.gatherApplicationResourceFiles();
        this.inspectApplicationResourceFiles(applicationResourceFiles);
        ArrayList<ResourceFile> resourceFiles = new ArrayList<ResourceFile>();
        resourceFiles.addAll(this.gatherFrameworkResourceFiles());
        resourceFiles.addAll(applicationResourceFiles);
        resourceFiles.add(this.generateManifest(resourceFiles));
        LOGGER.info("Copying " + resourceFiles.size() + " resource files");
        ArrayList<ResourceFile> textFiles = new ArrayList<ResourceFile>();
        List<JavaScriptLibrary> jsLibraries = this.copyJavaScriptLibraries();
        for (ResourceFile file : resourceFiles) {
            if (this.isFileType(file, TEXT_FILE_TYPES)) {
                textFiles.add(file);
                continue;
            }
            this.copyBinaryResourceFile(file);
        }
        this.rewriteHTML(textFiles, jsLibraries);
    }

    private List<JavaScriptLibrary> copyJavaScriptLibraries() {
        return JS_LIST.readLines(StandardCharsets.UTF_8).stream().filter(line -> !line.isEmpty() && !line.startsWith("#")).map(line -> this.copyJavaScriptLibrary((String)line)).toList();
    }

    private JavaScriptLibrary copyJavaScriptLibrary(String entry) {
        List entryParts = JS_LIB_SPLITTER.splitToList((CharSequence)entry);
        String url = (String)entryParts.getLast();
        String importAlias = entryParts.size() == 2 ? (String)entryParts.getFirst() : null;
        JavaScriptLibrary lib = new JavaScriptLibrary(url, importAlias);
        try {
            LOGGER.info("Downloading " + url);
            HttpResponse response = URLLoader.get((String)url);
            File outputFile = new File(this.outputDir, lib.getOutputFilePath());
            outputFile.getParentFile().mkdirs();
            Files.writeString(outputFile.toPath(), (CharSequence)response.body(), new OpenOption[0]);
            return lib;
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to download " + url, e);
        }
    }

    private boolean isFileType(ResourceFile needle, List<String> haystack) {
        return haystack.stream().anyMatch(type -> needle.getName().toLowerCase().endsWith((String)type));
    }

    private void rewriteHTML(List<ResourceFile> textFiles, List<JavaScriptLibrary> libs) {
        File outputFile = this.getOutputFile(INDEX_FILE);
        try (PrintWriter writer = new PrintWriter(outputFile, StandardCharsets.UTF_8.displayName());){
            for (String line : INDEX_FILE.readLines(StandardCharsets.UTF_8)) {
                line = line.replace("{project}", this.projectName);
                line = line.replace("{js-libraries}", this.generateScriptTags(libs));
                line = line.replace("{teavm-js-file}", SCRIPT_FILE_NAME);
                line = line.replace("{timestamp}", this.generateTimestampTag());
                line = line.replace("{build-id}", this.buildId);
                if ((line = line.replace("{meta}", this.generateMetaTags())).trim().equals("{resources}")) {
                    line = this.generateTextResourceFilesHTML(textFiles);
                }
                writer.println(line);
            }
        }
        catch (IOException e) {
            throw new MediaException("Cannot write to file: " + outputFile.getAbsolutePath(), e);
        }
    }

    private String generateScriptTags(List<JavaScriptLibrary> libs) {
        String html = libs.stream().filter(lib -> lib.importAlias == null).map(lib -> "<script src=\"" + lib.getOutputFilePath() + "\"></script>\n").collect(Collectors.joining(""));
        String importMapTemplate = "<script type=\"importmap\">\n    {\n        \"imports\": {\n            {importmap}\n        }\n    }\n</script>\n".trim();
        String importMapEntries = libs.stream().filter(lib -> lib.importAlias != null && !lib.importAlias.isEmpty()).map(lib -> "\"" + lib.importAlias + "\": \"./" + lib.getOutputFilePath() + "\"").collect(Collectors.joining(",\n            "));
        return html + importMapTemplate.replace("{importmap}", importMapEntries);
    }

    private String generateTimestampTag() {
        String timestamp = DateParser.format((Date)new Date(), (String)"yyyy-MM-dd HH:mm");
        return "<!-- Generated by Colorize MultimediaLib @ " + timestamp + " -->";
    }

    private String generateMetaTags() {
        if (this.meta == null || this.meta.isEmpty()) {
            return "";
        }
        return Splitter.on((String)",").splitToStream((CharSequence)this.meta).map(this::generateMetaTag).collect(Collectors.joining("\n"));
    }

    private String generateMetaTag(String entry) {
        List parts = Splitter.on((CharMatcher)CharMatcher.anyOf((CharSequence)"=:")).splitToList((CharSequence)entry);
        if (parts.size() != 2) {
            throw new CommandLineInterfaceException("Invalid meta tag: " + entry);
        }
        return String.format("<meta name=\"%s\" content=\"%s\" />", parts.get(0), parts.get(1));
    }

    private String generateTextResourceFilesHTML(List<ResourceFile> files) {
        StringBuilder buffer = new StringBuilder();
        for (ResourceFile file : files) {
            String id = this.normalizeFileName(file).replace(".", "_");
            String contents = file.read(StandardCharsets.UTF_8);
            buffer.append("<div id=\"" + id + "\">");
            buffer.append(XmlEscapers.xmlContentEscaper().escape(contents));
            buffer.append("</div>\n");
        }
        return buffer.toString();
    }

    private void copyBinaryResourceFile(ResourceFile file, File outputFile) {
        try (InputStream stream = file.openStream();){
            byte[] contents = stream.readAllBytes();
            Files.write(outputFile.toPath(), contents, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Cannot copy file: " + String.valueOf(file), e);
        }
    }

    private void copyBinaryResourceFile(ResourceFile file) {
        File outputFile = this.getOutputFile(file);
        this.copyBinaryResourceFile(file, outputFile);
    }

    private List<ResourceFile> gatherFrameworkResourceFiles() {
        return RESOURCES_LIST.readLines(StandardCharsets.UTF_8).stream().filter(line -> !line.isEmpty() && !line.startsWith("#")).map(ResourceFile::new).toList();
    }

    private List<ResourceFile> gatherApplicationResourceFiles() {
        try {
            return Files.walk(this.resourceDir.toPath(), new FileVisitOption[0]).map(path -> path.toFile()).filter(file -> !file.isDirectory() && !file.getName().startsWith(".")).filter(file -> !file.getAbsolutePath().contains("/lib/")).map(ResourceFile::new).toList();
        }
        catch (IOException e) {
            throw new MediaException("Cannot read resource files directory: " + String.valueOf(this.resourceDir), e);
        }
    }

    private ResourceFile generateManifest(List<ResourceFile> resourceFiles) {
        try {
            File tempDir = Files.createTempDirectory("resource-file-manifest", new FileAttribute[0]).toFile();
            File manifestFile = new File(tempDir, "resource-file-manifest");
            List<String> entries = resourceFiles.stream().map(file -> this.normalizeFileName((ResourceFile)file)).filter(file -> !file.endsWith(".js")).distinct().sorted().toList();
            Files.write(manifestFile.toPath(), entries, StandardCharsets.UTF_8, new OpenOption[0]);
            return new ResourceFile(manifestFile);
        }
        catch (IOException e) {
            throw new MediaException("Cannot generate resource file manifest", e);
        }
    }

    private File getOutputFile(ResourceFile file) {
        if (this.isFileType(file, RESOURCE_FILE_TYPES)) {
            File resourcesDir = new File(this.outputDir, "resources");
            resourcesDir.mkdir();
            return new File(resourcesDir, this.normalizeFileName(file));
        }
        return new File(this.outputDir, this.normalizeFileName(file));
    }

    protected File getScriptFile() {
        return new File(this.outputDir, SCRIPT_FILE_NAME);
    }

    private String normalizeFileName(ResourceFile file) {
        return file.getName().replace("/", "_");
    }

    private void cleanOldScripts() throws IOException {
        for (File file : FileUtils.walkFiles((File)this.outputDir, f -> f.getName().startsWith("script-"))) {
            FileUtils.delete((File)file);
        }
    }

    private void inspectApplicationResourceFiles(List<ResourceFile> resourceFiles) {
        List<String> fileNames = resourceFiles.stream().map(ResourceFile::getName).toList();
        for (String expected : EXPECTED_RESOURCES) {
            if (fileNames.contains(expected)) continue;
            LOGGER.warning("Missing resource file " + expected);
        }
    }

    private record JavaScriptLibrary(String url, String importAlias) {
        public String getOutputFilePath() {
            String fileName = (String)PATH_SPLITTER.splitToList((CharSequence)this.url).getLast();
            if (this.importAlias == null || this.importAlias.isEmpty()) {
                return "libraries/" + fileName;
            }
            List importPath = PATH_SPLITTER.splitToList((CharSequence)this.importAlias);
            List subDirs = importPath.subList(0, importPath.size() - 1);
            ArrayList<String> outputFilePath = new ArrayList<String>();
            outputFilePath.add("libraries");
            outputFilePath.addAll(subDirs);
            outputFilePath.add(fileName);
            return String.join((CharSequence)"/", outputFilePath);
        }
    }
}

