<link rev="doc" href="../volume-breaking.scss"/>

~~~sass
/**
 * CSS module with default volume breaking rules.
 *
 * In order for this style sheet to work,
 * http://www.daisy.org/pipeline/modules/braille/dtbook-to-pef/xsl/volume-breaking.xsl
 * must be included before it in the list of style sheets.
 */

/*
 * Don't split "leaf sections" across volumes.
 *
 * A leaf section is a block of adjacent content with the same sectioning depth.
 * Which means:
 * - either; a level element with no child levels
 * - or; the content in a level (<h1>, <p>, etc.), which may have level element siblings
 *
 * If a level contains other levels, then volume-breaking.xsl wraps the normal content
 * (<h1>, <p>, etc.) in the level in a <div class="leaf-section"> element.
 */

/**
 * Amount with which volume breaks are allowed within "leaf sections", meaning that a
 * volume is not broken right before a heading.
 *
 * The value can range from 1 to 9. Setting a higher value results in volumes that are
 * more equal in size.
 */
$allow-volume-break-inside-leaf-section-factor: 9;

/**
 * Amount with which preference is given to volume breaks before higher level headings (h1
 * is the highest level) in comparison to lower level headings.
 *
 * The value can range from 0 to ? and can be a non-integer. A value of 0 means no
 * preference is given. A higher value results in volumes that are less equal in
 * size. Depending on the value of $allow-volume-break-inside-leaf-section-factor,
 * increasing $prefer-volume-break-before-higher-level-factor beyond a certain value does
 * not have an effect anymore. The default value is 1.
 */
$prefer-volume-break-before-higher-level-factor: 0;

/* ==================================================================================== */

$min-volume-keep-priority: min(9, max(1, round($allow-volume-break-inside-leaf-section-factor)));
@for $level from 1 through 6 {
    $volume-keep-priority: min(10,
                               max($min-volume-keep-priority,
                                   round(10 - $level * $prefer-volume-break-before-higher-level-factor)));
    @if $volume-keep-priority < 10 {
        h#{$level} {
            volume-break-inside: -obfl-keep($volume-keep-priority);
        }
    }
}
level1:not(:has(> level2, div.leaf-section)),
level2:not(:has(> level3, div.leaf-section)),
level3:not(:has(> level4, div.leaf-section)),
level4:not(:has(> level5, div.leaf-section)),
level5:not(:has(> level6, div.leaf-section)),
level6:not(:has(div.leaf-section)),
div.leaf-section {
    volume-break-inside: -obfl-keep($min-volume-keep-priority);
}
~~~
