package nl.dedicon.pipeline.braille.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.net.URI;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import java.util.Arrays;
import java.util.Locale;

import org.daisy.common.file.URIs;
import org.daisy.common.file.URLs;
import org.daisy.pipeline.braille.common.AbstractBrailleTranslator;
import org.daisy.pipeline.braille.common.AbstractTransformProvider;
import org.daisy.pipeline.braille.common.AbstractTransformProvider.util.Function;
import org.daisy.pipeline.braille.common.AbstractTransformProvider.util.Iterables;
import static org.daisy.pipeline.braille.common.AbstractTransformProvider.util.Iterables.concat;
import static org.daisy.pipeline.braille.common.AbstractTransformProvider.util.Iterables.transform;
import static org.daisy.pipeline.braille.common.AbstractTransformProvider.util.logCreate;
import static org.daisy.pipeline.braille.common.AbstractTransformProvider.util.logSelect;
import org.daisy.pipeline.braille.common.BrailleTranslator;
import org.daisy.pipeline.braille.common.BrailleTranslatorProvider;
import org.daisy.pipeline.braille.common.CSSStyledText;
import org.daisy.pipeline.braille.common.Hyphenator;
import org.daisy.pipeline.braille.common.HyphenatorProvider;
import org.daisy.pipeline.braille.common.Query;
import org.daisy.pipeline.braille.common.Query.Feature;
import org.daisy.pipeline.braille.common.Query.MutableQuery;
import static org.daisy.pipeline.braille.common.Query.util.mutableQuery;
import static org.daisy.pipeline.braille.common.Query.util.query;
import org.daisy.pipeline.braille.common.TransformProvider;
import static org.daisy.pipeline.braille.common.TransformProvider.util.dispatch;
import static org.daisy.pipeline.braille.common.TransformProvider.util.memoize;
import static org.daisy.pipeline.braille.common.util.Locales.parseLocale;
import org.daisy.pipeline.braille.liblouis.LiblouisTranslator;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.osgi.service.component.ComponentContext;

public interface DediconTranslator {

    @Component(
            name = "nl.dedicon.pipeline.braille.impl.DediconTranslator.Provider",
            service = {
                TransformProvider.class,
                BrailleTranslatorProvider.class
            }
    )
    public class Provider extends AbstractTransformProvider<BrailleTranslator> implements BrailleTranslatorProvider<BrailleTranslator> {

        private URI href;

        @Activate
        void activate(final Map<?, ?> properties) {
            href = URIs.asURI(URLs.getResourceFromJAR("/xml/xproc/block-translate.xpl", DediconTranslator.class));
        }

        private final static Iterable<BrailleTranslator> empty = Iterables.<BrailleTranslator>empty();

        private final static List<String> supportedInput = ImmutableList.of("css", "text-css", "dtbook", "html");
        private final static List<String> supportedOutput = ImmutableList.of("css", "braille");

        /**
         * Recognized features:
         *
         * - translator: Will only match if the value is `dedicon'.
         * - locale:     Used for selecting hyphenator.
         *
         */
        @Override
        protected final Iterable<BrailleTranslator> _get(Query query) {
            // @todo determine liblouisTable based on query parameter locale
            final Query liblouisTable = query("(locale:nl_NL)");
            Query hyphenationLocale = null;
            
            final MutableQuery q = mutableQuery(query);
            for (Feature f : q.removeAll("input")) {
                if (!supportedInput.contains(f.getValue().get())) {
                    return empty;
                }
            }
            for (Feature f : q.removeAll("output")) {
                if (!supportedOutput.contains(f.getValue().get())) {
                    return empty;
                }
            }
            if (q.containsKey("locale")) {
                hyphenationLocale = mutableQuery().addAll(q.removeAll("locale"));
            }
            if (hyphenationLocale == null) {
                return empty;
            }
            if (q.containsKey("translator")) {
                if ("dedicon".equals(q.removeOnly("translator").getValue().get())) {
                    if (q.isEmpty()) {

                        Query hyphenTable = mutableQuery(hyphenationLocale).add("hyphenator", "hyphen");
                        Query fallbackHyphenationTable = mutableQuery(hyphenationLocale).add("hyphenator", "tex");

                        Iterable<Hyphenator> hyphenators = concat(
                                logSelect(hyphenTable, hyphenatorProvider),
                                logSelect(fallbackHyphenationTable, hyphenatorProvider)
                        );
                        return concat(
                                transform(
                                        concat(
                                                transform(
                                                        hyphenators,
                                                        new Function<Hyphenator, String>() {
                                                            @Override
                                                            public String _apply(Hyphenator h) {
                                                                return h.getIdentifier();
                                                            }
                                                        }
                                                ),
                                                "liblouis"
                                        ),
                                        new Function<String, Iterable<BrailleTranslator>>() {
                                            @Override
                                            public Iterable<BrailleTranslator> _apply(String hyphenator) {
                                                final Query translatorQuery = mutableQuery(liblouisTable).add("hyphenator", hyphenator);
                                                return transform(
                                                        logSelect(translatorQuery, liblouisTranslatorProvider),
                                                        new Function<LiblouisTranslator, BrailleTranslator>() {
                                                            @Override
                                                            public BrailleTranslator _apply(LiblouisTranslator translator) {
                                                                return __apply(logCreate(new TransformImpl(translatorQuery.toString(), translator)));
                                                            }
                                                        }
                                                );
                                            }
                                        }
                                )
                        );
                    }
                }
            }
            return empty;
        }

        private class TransformImpl extends AbstractBrailleTranslator {

            private final FromStyledTextToBraille translator;
            private final XProc xproc;

            private TransformImpl(String translatorQuery, LiblouisTranslator translator) {
                Map<String, String> options = ImmutableMap.of("text-transform", mutableQuery().add("id", this.getIdentifier()).toString());
                xproc = new XProc(href, null, options);
                this.translator = translator.fromStyledTextToBraille();
            }

            @Override
            public XProc asXProc() {
                return xproc;
            }

            @Override
            public FromStyledTextToBraille fromStyledTextToBraille() {
                return fromStyledTextToBraille;
            }

            private final FromStyledTextToBraille fromStyledTextToBraille = new FromStyledTextToBraille() {
                @Override
                public java.lang.Iterable<String> transform(java.lang.Iterable<CSSStyledText> styledText, int from, int to) {
                    return translator.transform(styledText, from, to);
                }
            };

            @Override
            public String toString() {
                return MoreObjects.toStringHelper(DediconTranslator.class.getSimpleName())
                        .add("id", getIdentifier())
                        .add("liblouisTranslator", translator)
                        .toString();
            }
        }

        @Reference(
                name = "LiblouisTranslatorProvider",
                unbind = "unbindLiblouisTranslatorProvider",
                service = LiblouisTranslator.Provider.class,
                cardinality = ReferenceCardinality.MULTIPLE,
                policy = ReferencePolicy.DYNAMIC
        )
        protected void bindLiblouisTranslatorProvider(LiblouisTranslator.Provider provider) {
            liblouisTranslatorProviders.add(provider);
        }

        protected void unbindLiblouisTranslatorProvider(LiblouisTranslator.Provider provider) {
            liblouisTranslatorProviders.remove(provider);
            liblouisTranslatorProvider.invalidateCache();
        }

        private final List<TransformProvider<LiblouisTranslator>> liblouisTranslatorProviders
                = new ArrayList<>();
        private final TransformProvider.util.MemoizingProvider<LiblouisTranslator> liblouisTranslatorProvider
                = memoize(dispatch(liblouisTranslatorProviders));

        @Reference(
                name = "HyphenatorProvider",
                unbind = "unbindHyphenatorProvider",
                service = HyphenatorProvider.class,
                cardinality = ReferenceCardinality.MULTIPLE,
                policy = ReferencePolicy.DYNAMIC
        )
        @SuppressWarnings(
                "unchecked" // safe cast to TransformProvider<Hyphenator>
        )
        protected void bindHyphenatorProvider(HyphenatorProvider<?> provider) {
            hyphenatorProviders.add((TransformProvider<Hyphenator>) provider);
        }

        protected void unbindHyphenatorProvider(HyphenatorProvider<?> provider) {
            hyphenatorProviders.remove((HyphenatorProvider)provider);
            hyphenatorProvider.invalidateCache();
        }

        private final List<TransformProvider<Hyphenator>> hyphenatorProviders
                = new ArrayList<>();
        private final TransformProvider.util.MemoizingProvider<Hyphenator> hyphenatorProvider
                = memoize(dispatch(hyphenatorProviders));

    }
}
