/*
 * Copyright 2021 Elements
 *
 * Licensed under the BSD-3 license:
 * https://opensource.org/licenses/BSD-3-Clause
 */
package nl.elements.mobilization.logging

import android.os.Build
import android.util.Log
import com.google.firebase.crashlytics.FirebaseCrashlytics
import timber.log.Timber
import java.util.regex.Pattern

/*
 * Original License as derived from https://github.com/chrisbanes/tivi/
 * These classes are copied and partially modified
 * Copyright 2019 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
@Suppress("TooManyFunctions")
class ElementsLogger constructor(
    private val firebaseCrashlytics: FirebaseCrashlytics
) : Logger {
    fun setup(debugMode: Boolean, logsToCrashlytics: Boolean) {
        if (debugMode) {
            Timber.plant(ReportingDebugTree())
        }

        if (logsToCrashlytics) {
            Timber.plant(CrashlyticsTree(firebaseCrashlytics))
        }
    }

    override fun setCustomKey(key: String, value: String) {
        firebaseCrashlytics.setCustomKey(key, value)
    }

    override fun setUserId(id: String) {
        firebaseCrashlytics.setUserId(id)
    }

    override fun v(message: String, vararg args: Any?) = Timber.v(message, *args)

    override fun v(t: Throwable, message: String, vararg args: Any?) = Timber.v(t, message, *args)

    override fun v(t: Throwable) = Timber.v(t)

    override fun d(message: String, vararg args: Any?) = Timber.d(message, *args)

    override fun d(t: Throwable, message: String, vararg args: Any?) = Timber.d(t, message, *args)

    override fun d(t: Throwable) = Timber.d(t)

    override fun i(message: String, vararg args: Any?) = Timber.i(message, *args)

    override fun i(t: Throwable, message: String, vararg args: Any?) = Timber.i(t, message, *args)

    override fun i(t: Throwable) = Timber.i(t)

    override fun w(message: String, vararg args: Any?) = Timber.w(message, *args)

    override fun w(t: Throwable, message: String, vararg args: Any?) = Timber.w(t, message, *args)

    override fun w(t: Throwable) = Timber.w(t)

    override fun e(message: String, vararg args: Any?) = Timber.e(message, *args)

    override fun e(t: Throwable, message: String, vararg args: Any?) = Timber.e(t, message, *args)

    override fun e(t: Throwable) = Timber.e(t)

    override fun wtf(message: String, vararg args: Any?) = Timber.wtf(message, *args)

    override fun wtf(t: Throwable, message: String, vararg args: Any?) =
        Timber.wtf(t, message, *args)

    override fun wtf(t: Throwable) = Timber.wtf(t)
}

/**
 * Special version of [Timber.DebugTree] which is tailored for Timber being wrapped
 * within another class.
 *
 * Copied & modified from https://github.com/chrisbanes/tivi/
 */
private class ReportingDebugTree : Timber.DebugTree() {
    override fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
        super.log(priority, createClassTag(), message, t)
    }

    private fun createClassTag(): String {
        val stackTrace = Throwable().stackTrace
        if (stackTrace.size <= CALL_STACK_INDEX) {
            throw IllegalStateException("Synthetic stacktrace didn't have enough elements: are you using proguard?")
        }
        var tag = stackTrace[CALL_STACK_INDEX].className
        val m = ANONYMOUS_CLASS.matcher(tag)
        if (m.find()) {
            tag = m.replaceAll("")
        }
        tag = tag.substring(tag.lastIndexOf('.') + 1)
        // Tag length limit was removed in API 24.
        return if (tag.length <= MAX_TAG_LENGTH || Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            tag
        } else tag.substring(0, MAX_TAG_LENGTH)
    }

    companion object {
        private const val MAX_TAG_LENGTH = 23
        private const val CALL_STACK_INDEX = 7
        private val ANONYMOUS_CLASS = Pattern.compile("(\\$\\d+)+$")
    }
}

/**
 * Copied & modified from https://github.com/chrisbanes/tivi/
 */
private class CrashlyticsTree(
    private val firebaseCrashlytics: FirebaseCrashlytics
) : Timber.Tree() {
    override fun isLoggable(tag: String?, priority: Int): Boolean = priority > Log.DEBUG

    override fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
        firebaseCrashlytics.log(message)

        if (t != null) {
            firebaseCrashlytics.recordException(t)
        }
    }
}
