/*
 * Copyright (c) 2023 Guus der Kinderen. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nl.goodbytes.xmpp.xep0363.clamav;

import nl.goodbytes.xmpp.xep0363.MalwareDetectedException;
import nl.goodbytes.xmpp.xep0363.MalwareScanner;
import nl.goodbytes.xmpp.xep0363.RepositoryManager;
import nl.goodbytes.xmpp.xep0363.SecureUniqueId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xyz.capybara.clamav.ClamavClient;
import xyz.capybara.clamav.ClamavException;
import xyz.capybara.clamav.commands.scan.result.ScanResult;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;

public class ClamavMalwareScanner implements MalwareScanner
{
    private static final Logger Log = LoggerFactory.getLogger(ClamavMalwareScanner.class);

    private final String hostname;

    private final int port;

    private final Duration connectTimeout;

    private ClamavClient client;

    public ClamavMalwareScanner(final String hostname, final int port, final Duration connectTimeout)
    {
        this.hostname = hostname;
        this.port = port;
        this.connectTimeout = connectTimeout.toMillis() > Integer.MAX_VALUE ? Duration.ofMillis(Integer.MAX_VALUE) : connectTimeout;
    }

    @Override
    public synchronized void initialize() throws IOException
    {
        client = new ClamavClient(hostname, port);
        if (!client.isReachable((int) connectTimeout.toMillis())) {
            throw new IOException("Clamav daemon not reachable on " + hostname + ":" + port);
        }

        try {
            final String version = client.version();
            Log.info("Successfully connected to Clamav daemon " + version + ".");
        } catch (Throwable t) {
            Log.debug("Unable to determine Clamav daemon version.");
            Log.info("Successfully connected to Clamav daemon!");
        }
    }

    @Override
    public synchronized void destroy()
    {
    }

    @Override
    public void scan(final SecureUniqueId uuid) throws MalwareDetectedException, IOException
    {
        synchronized (this) {
            try {
                client.ping();
            } catch (ClamavException e) {
                Log.info("Unsuccessful ping of the Clamav daemon. Trying to re-initialize the client.", e);
                initialize();
            }
        }

        try (final InputStream is = RepositoryManager.getInstance().getRepository().getInputStream(uuid)) {
            final ScanResult scanResult = client.scan(is);
            if (!(scanResult instanceof ScanResult.OK)) {
                if (scanResult instanceof ScanResult.VirusFound) {
                    ((ScanResult.VirusFound) scanResult).getFoundViruses().values()
                        .forEach(malware -> Log.warn("Detected malware in slot '{}': {}", uuid, malware));
                }
                throw new MalwareDetectedException();
            }
        }
    }
}
