/*
 * Copyright (c) 2015 Mark Platvoet<mplatvoet@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * THE SOFTWARE.
 */

package nl.mplatvoet.komponents.kovenant.android

import nl.mplatvoet.komponents.kovenant.Dispatcher
import nl.mplatvoet.komponents.kovenant.Kovenant
import nl.mplatvoet.komponents.kovenant.buildDispatcher

/**
 * Configures Kovenant for common Android scenarios.
 *
 * @return `Disposable` to properly shutdown Kovenant
 */
public fun configureKovenant(): Disposable {
    val callbackDispatcher = buildDispatcher {
        name = "kovenant-callback"
        numberOfThreads = 1

        configurePollStrategy {
            yielding(numberOfPolls = 100)
            blocking()
        }
    }
    val workerDispatcher = buildDispatcher {
        name = "kovenant-worker"

        configurePollStrategy {
            yielding(numberOfPolls = 100)
            blocking()
        }
    }

    Kovenant.configure {
        this.callbackDispatcher = callbackDispatcher
        this.workerDispatcher = workerDispatcher
    }
    return DispatchersDisposable(workerDispatcher, callbackDispatcher)
}


/**
 * Disposes of a resource.
 *
 */
public trait Disposable {
    fun close(force: Boolean = false)
}

private class DispatchersDisposable(private vararg val dispatcher: Dispatcher) : Disposable {
    override fun close(force: Boolean) {
        dispatcher forEach {
            close(force, it)
        }
    }

    private fun close(force: Boolean, dispatcher: Dispatcher) {
        try {
            if (force) {
                dispatcher.stop(force = true)
            } else {
                dispatcher.stop(block = true)
            }
        } catch(e: Exception) {
            //ignore, nothing we can do
        }
    }

}

