package eu.woolplatform.utils.i18n;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import eu.woolplatform.utils.io.ResourceLocator;

/**
 * This class can find a preferred string map for a specific language. After
 * construction you can set properties with the set methods. Then call {@link
 * #find() find()}.
 *
 * <p><b>Example</b></p>
 *
 * <p>Find properties:</p>
 *
 * <p><ul>
 * <li>userLocales: ["nl_NL"]</li>
 * <li>honorifics: true</li>
 * </ul></p>
 *
 * <p>It will try the following keys, in this order:</p>
 *
 * <p><ul>
 * <li>"nl_NL_v" (if the locale contains a country; v = vos, t = tu)</li>
 * <li>"nl_NL" (if the locale contains a country)</li>
 * <li>"nl_v"</li>
 * <li>"nl"</li>
 * <li>"v"</li>
 * <li>""</li>
 * </ul></p>
 *
 * <p>If nothing is found, it will try the following locales: en_GB, en_US,
 * en.</p>
 *
 * @author Dennis Hofs (RRD)
 */
public class I18nStringMapFinder {
	private I18nResourceFinder finder;
	private Map<String,Map<String,String>> stringMaps;

	public I18nStringMapFinder(Map<String,Map<String,String>> stringMaps) {
		this.stringMaps = stringMaps;
		finder = new I18nResourceFinder("");
		finder.setExtension(null);
		finder.setResourceLocator(new StringMapLocator());
	}

	/**
	 * Sets the preferred locale of the user. If no resource is found for this
	 * locale, the resource finder will try en_GB, en_US or en. The default is
	 * the locale of the system.
	 *
	 * @param userLocale the preferred locale of the user
	 * @see #setUserLocales(List)
	 */
	public void setUserLocale(Locale userLocale) {
		finder.setUserLocale(userLocale);
	}

	/**
	 * Sets the preferred locales of the user. If no resource is found for
	 * these locales, the resource finder will try en_GB, en_US or en. The
	 * default is the locale of the system.
	 *
	 * @param userLocales the preferred locales of the user (at least one)
	 */
	public void setUserLocales(List<Locale> userLocales) {
		finder.setUserLocales(userLocales);
	}

	/**
	 * Sets whether the resource should use honorifics. This is true for vos
	 * (u, vous, Sie) in tu-vos distinction, and false for tu (jij, tu, du).
	 * For languages without honorifics, such as English, there will be no
	 * resources with tu-vos designation, so the value of this property is not
	 * relevant. The default is true.
	 *
	 * @param honorifics true if the resource should use honorifics, false
	 * otherwise
	 */
	public void setHonorifics(boolean honorifics) {
		finder.setHonorifics(honorifics);
	}

	/**
	 * Tries to find a string map matching the specified properties. If a map is
	 * found, this method will return that map. Otherwise it returns an empty
	 * map.
	 *
	 * @return the string map
	 */
	public Map<String,String> find() {
		if (!finder.find())
			return new LinkedHashMap<>();
		return stringMaps.get(finder.getName());
	}

	private class StringMapLocator implements ResourceLocator {
		@Override
		public boolean resourceExists(String path) {
			return stringMaps.containsKey(path);
		}

		@Override
		public InputStream openResource(String path) throws IOException {
			return null;
		}
	}
}
