/*
 * Decompiled with CFR 0.152.
 */
package nl.sidnlabs.pcap.decoder;

import com.google.common.collect.Multimap;
import com.google.common.collect.TreeMultimap;
import com.google.common.primitives.Bytes;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import nl.sidnlabs.pcap.PcapReaderUtil;
import nl.sidnlabs.pcap.decoder.Decoder;
import nl.sidnlabs.pcap.decoder.ICMPDecoder;
import nl.sidnlabs.pcap.packet.DNSPacket;
import nl.sidnlabs.pcap.packet.Datagram;
import nl.sidnlabs.pcap.packet.DatagramPayload;
import nl.sidnlabs.pcap.packet.ICMPPacket;
import nl.sidnlabs.pcap.packet.Packet;
import nl.sidnlabs.pcap.packet.PacketFactory;
import nl.sidnlabs.pcap.util.IPv4Util;
import nl.sidnlabs.pcap.util.IPv6Util;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class IPDecoder {
    private static final Logger log = LogManager.getLogger(IPDecoder.class);
    public static final int IP_PROTOCOL_VERSION_4 = 4;
    public static final int IP_PROTOCOL_VERSION_6 = 6;
    public static final int IP_TOTAL_LEN_OFFSET = 2;
    public static final int IP_FLAGS = 6;
    public static final int IP_FRAGMENT_OFFSET = 6;
    private Multimap<Datagram, DatagramPayload> datagrams = TreeMultimap.create();
    private Decoder tcpReader;
    private Decoder udpReader;
    private ICMPDecoder icmpDecoder;

    public IPDecoder(Decoder tcpReader, Decoder udpReader, ICMPDecoder icmpDecoder) {
        this.tcpReader = tcpReader;
        this.udpReader = udpReader;
        this.icmpDecoder = icmpDecoder;
    }

    public void printStats() {
        this.udpReader.printStats();
        if (this.tcpReader != null) {
            this.tcpReader.printStats();
        }
        this.icmpDecoder.printStats();
    }

    public Packet decode(byte[] packetData, int ipStart, long packetTimestampSecs, long packetTimestampMicros) {
        if (ipStart == -1) {
            return Packet.NULL;
        }
        Packet p = this.createPacket(packetData, ipStart);
        p.setTsSec(packetTimestampSecs);
        p.setTsMicro(packetTimestampMicros);
        p.setTsMilli(packetTimestampSecs * 1000L + (long)Math.round((float)packetTimestampMicros / 1000.0f));
        return this.decode(p, packetData, ipStart);
    }

    private Packet decode(Packet packet, byte[] packetData, int ipStart) {
        int ipHeaderLen;
        int ipProtocolHeaderVersion = IPv4Util.getInternetProtocolHeaderVersion(packetData, ipStart);
        packet.setIpVersion(ipProtocolHeaderVersion);
        int totalLength = 0;
        if (ipProtocolHeaderVersion == 4) {
            ipHeaderLen = IPv4Util.getInternetProtocolHeaderLength(packetData, ipStart);
            packet.setIpHeaderLen(ipHeaderLen);
            packet.setTtl(IPv4Util.decodeTTL(packetData, ipStart));
            packet.setSrcAddr(IPv4Util.decodeSrc(packetData, ipStart));
            packet.setDstAddr(IPv4Util.decodeDst(packetData, ipStart));
            if (packet.getSrcAddr() != null) {
                packet.setSrc(packet.getSrcAddr().getHostAddress());
            }
            if (packet.getDstAddr() != null) {
                packet.setDst(packet.getDstAddr().getHostAddress());
            }
            packet.setIpId(IPv4Util.decodeId(packetData, ipStart));
            totalLength = PcapReaderUtil.convertShort(packetData, ipStart + 2);
            this.decodeV4Fragmented(packet, ipStart, packetData);
        } else {
            ipHeaderLen = IPv6Util.getInternetProtocolHeaderLength(packetData, ipStart);
            packet.setIpHeaderLen(ipHeaderLen);
            packet.setTtl(IPv6Util.decodeTTL(packetData, ipStart));
            packet.setSrcAddr(IPv6Util.decodeSrc(packetData, ipStart));
            packet.setDstAddr(IPv6Util.decodeDst(packetData, ipStart));
            if (packet.getSrcAddr() != null) {
                packet.setSrc(packet.getSrcAddr().getHostAddress());
            }
            if (packet.getDstAddr() != null) {
                packet.setDst(packet.getDstAddr().getHostAddress());
            }
            packet.setIpId(IPv6Util.decodeId(packetData, ipStart));
            int payloadLength = PcapReaderUtil.convertShort(packetData, ipStart + 4);
            totalLength = payloadLength + 40;
            this.decodeV6Fragmented(packet, ipStart, packetData);
            if (packet.isFragmented()) {
                IPv6Util.buildInternetProtocolV6ExtensionHeaderFragment(packet, packetData, ipStart);
            }
        }
        packet.setTotalLength(totalLength);
        int padding = Math.max(packetData.length - (totalLength + ipStart), 0);
        packetData = Arrays.copyOfRange(packetData, ipStart + packet.getIpHeaderLen(), packetData.length - padding);
        byte[] reassembledData = this.reassemble(packet, packetData);
        if (reassembledData.length == 0) {
            return Packet.NULL;
        }
        return this.handlePayload(packet, reassembledData);
    }

    public Packet createPacket(byte[] packetData, int ipStart) {
        int ipProtocolHeaderVersion = IPv4Util.getInternetProtocolHeaderVersion(packetData, ipStart);
        byte protocol = -1;
        if (ipProtocolHeaderVersion == 4) {
            protocol = IPv4Util.decodeProtocol(packetData, ipStart);
        } else if (ipProtocolHeaderVersion == 6) {
            protocol = IPv6Util.decodeProtocol(packetData, ipStart);
        } else {
            log.error("Unsupported IP version " + ipProtocolHeaderVersion + " ipstart=" + ipStart);
            return Packet.NULL;
        }
        return PacketFactory.create(protocol);
    }

    private Packet handlePayload(Packet packet, byte[] packetData) {
        if (1 == packet.getProtocol() || 58 == packet.getProtocol()) {
            this.icmpDecoder.reassemble((ICMPPacket)packet, packetData);
            return packet;
        }
        if (6 == packet.getProtocol()) {
            if (this.tcpReader == null) {
                return Packet.NULL;
            }
            packet = this.tcpReader.reassemble(packet, packetData);
        } else if (17 == packet.getProtocol()) {
            packet = this.udpReader.reassemble(packet, packetData);
        }
        if (packet instanceof DNSPacket && ((DNSPacket)packet).getMessageCount() == 0) {
            return Packet.NULL;
        }
        return packet;
    }

    private void decodeV6Fragmented(Packet packet, int ipStart, byte[] packetData) {
        byte nxtHdr = packetData[ipStart + 6];
        packet.setFragmented(nxtHdr == 44);
    }

    private void decodeV4Fragmented(Packet packet, int ipStart, byte[] packetData) {
        long fragmentOffset = (long)(PcapReaderUtil.convertShort(packetData, ipStart + 6) & 0x1FFF) * 8L;
        packet.setFragOffset(fragmentOffset);
        int flags = packetData[ipStart + 6] & 0xE0;
        if ((flags & 0x40) == 64) {
            packet.setDoNotFragment(true);
        } else if ((flags & 0x20) == 32 || fragmentOffset != 0L) {
            packet.setFragmented(true);
            packet.setLastFragment((flags & 0x20) == 0 && fragmentOffset != 0L);
        }
    }

    public byte[] reassemble(Packet packet, byte[] packetData) {
        if (!packet.isFragmented()) {
            return packetData;
        }
        Datagram datagram = packet.getDatagram();
        DatagramPayload payload = new DatagramPayload(packet.getFragOffset(), packetData);
        this.datagrams.put((Object)datagram, (Object)payload);
        if (packet.isLastFragment()) {
            byte[] reassembledPacketData = new byte[]{};
            Collection datagramPayloads = this.datagrams.removeAll((Object)datagram);
            if (datagramPayloads != null && !datagramPayloads.isEmpty()) {
                int reassembledFragments = 0;
                DatagramPayload prev = null;
                for (DatagramPayload datagramPayload : datagramPayloads) {
                    if (prev == null && datagramPayload.getOffset() != 0L) {
                        if (log.isDebugEnabled()) {
                            log.debug("Datagram chain not starting at 0. Probably received packets out-of-order. Can't reassemble this packet.");
                        }
                        return new byte[0];
                    }
                    if (prev != null && !datagramPayload.linked(prev)) {
                        if (log.isDebugEnabled()) {
                            log.debug("Broken datagram chain between " + datagramPayload + " and " + prev + ". Can't reassemble this packet.");
                        }
                        return new byte[0];
                    }
                    reassembledPacketData = Bytes.concat((byte[][])new byte[][]{reassembledPacketData, datagramPayload.getPayload()});
                    ++reassembledFragments;
                    prev = datagramPayload;
                }
                packet.setReassembledFragments(reassembledFragments);
            }
            return reassembledPacketData;
        }
        return new byte[0];
    }

    public Multimap<Datagram, DatagramPayload> getDatagrams() {
        return this.datagrams;
    }

    public void setDatagrams(Multimap<Datagram, DatagramPayload> datagrams) {
        this.datagrams = datagrams;
    }

    public void clearCache(int ipFragmentTTL) {
        long now = System.currentTimeMillis();
        List dgExpiredList = this.datagrams.keySet().stream().filter(k -> k.getTime() + (long)ipFragmentTTL <= now).collect(Collectors.toList());
        log.info("IP datagram cache size: " + this.datagrams.size());
        log.info("Expired (to be removed) IP datagrams: " + dgExpiredList.size());
        dgExpiredList.stream().forEach(dg -> this.datagrams.removeAll(dg));
    }

    public Decoder getTcpReader() {
        return this.tcpReader;
    }

    public Decoder getUdpReader() {
        return this.udpReader;
    }

    public ICMPDecoder getIcmpDecoder() {
        return this.icmpDecoder;
    }

    public void setTcpReader(Decoder tcpReader) {
        this.tcpReader = tcpReader;
    }

    public void setUdpReader(Decoder udpReader) {
        this.udpReader = udpReader;
    }

    public void setIcmpDecoder(ICMPDecoder icmpDecoder) {
        this.icmpDecoder = icmpDecoder;
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof IPDecoder)) {
            return false;
        }
        IPDecoder other = (IPDecoder)o;
        if (!other.canEqual(this)) {
            return false;
        }
        Multimap<Datagram, DatagramPayload> this$datagrams = this.getDatagrams();
        Multimap<Datagram, DatagramPayload> other$datagrams = other.getDatagrams();
        if (this$datagrams == null ? other$datagrams != null : !this$datagrams.equals(other$datagrams)) {
            return false;
        }
        Decoder this$tcpReader = this.getTcpReader();
        Decoder other$tcpReader = other.getTcpReader();
        if (this$tcpReader == null ? other$tcpReader != null : !this$tcpReader.equals(other$tcpReader)) {
            return false;
        }
        Decoder this$udpReader = this.getUdpReader();
        Decoder other$udpReader = other.getUdpReader();
        if (this$udpReader == null ? other$udpReader != null : !this$udpReader.equals(other$udpReader)) {
            return false;
        }
        ICMPDecoder this$icmpDecoder = this.getIcmpDecoder();
        ICMPDecoder other$icmpDecoder = other.getIcmpDecoder();
        return !(this$icmpDecoder == null ? other$icmpDecoder != null : !((Object)this$icmpDecoder).equals(other$icmpDecoder));
    }

    protected boolean canEqual(Object other) {
        return other instanceof IPDecoder;
    }

    public int hashCode() {
        int PRIME = 59;
        int result = 1;
        Multimap<Datagram, DatagramPayload> $datagrams = this.getDatagrams();
        result = result * 59 + ($datagrams == null ? 43 : $datagrams.hashCode());
        Decoder $tcpReader = this.getTcpReader();
        result = result * 59 + ($tcpReader == null ? 43 : $tcpReader.hashCode());
        Decoder $udpReader = this.getUdpReader();
        result = result * 59 + ($udpReader == null ? 43 : $udpReader.hashCode());
        ICMPDecoder $icmpDecoder = this.getIcmpDecoder();
        result = result * 59 + ($icmpDecoder == null ? 43 : ((Object)$icmpDecoder).hashCode());
        return result;
    }

    public String toString() {
        return "IPDecoder(datagrams=" + this.getDatagrams() + ", tcpReader=" + this.getTcpReader() + ", udpReader=" + this.getUdpReader() + ", icmpDecoder=" + this.getIcmpDecoder() + ")";
    }
}

