/*
 * Decompiled with CFR 0.152.
 */
package nl.sidnlabs.pcap.decoder;

import com.google.common.collect.Multimap;
import com.google.common.collect.TreeMultimap;
import com.google.common.primitives.Bytes;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import nl.sidnlabs.pcap.PcapReaderUtil;
import nl.sidnlabs.pcap.decoder.Decoder;
import nl.sidnlabs.pcap.decoder.ICMPDecoder;
import nl.sidnlabs.pcap.decoder.TCPDecoder;
import nl.sidnlabs.pcap.packet.DNSPacket;
import nl.sidnlabs.pcap.packet.Datagram;
import nl.sidnlabs.pcap.packet.DatagramPayload;
import nl.sidnlabs.pcap.packet.FlowData;
import nl.sidnlabs.pcap.packet.ICMPPacket;
import nl.sidnlabs.pcap.packet.Packet;
import nl.sidnlabs.pcap.packet.PacketFactory;
import nl.sidnlabs.pcap.packet.TCPFlow;
import nl.sidnlabs.pcap.util.IPv4Util;
import nl.sidnlabs.pcap.util.IPv6Util;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class IPDecoder {
    private static final Logger log = LogManager.getLogger(IPDecoder.class);
    public static final int IP_PROTOCOL_VERSION_4 = 4;
    public static final int IP_PROTOCOL_VERSION_6 = 6;
    public static final int IP_TOTAL_LEN_OFFSET = 2;
    public static final int IP_FLAGS = 6;
    public static final int IP_FRAGMENT_OFFSET = 6;
    private Multimap<Datagram, DatagramPayload> datagrams = TreeMultimap.create();
    private int counter;
    private int counterPayload;
    private Decoder tcpReader;
    private Decoder udpReader;
    private ICMPDecoder icmpDecoder;
    private long lastPacketTs;

    public IPDecoder(Decoder tcpReader, Decoder udpReader, ICMPDecoder icmpDecoder) {
        this.tcpReader = tcpReader;
        this.udpReader = udpReader;
        this.icmpDecoder = icmpDecoder;
    }

    public void printStats() {
        log.info("------------- IP Decoder Stats --------------------------");
        log.info("Packets total: {}", (Object)this.counter);
        log.info("Packets payload: {}", (Object)this.counterPayload);
        this.udpReader.printStats();
        if (this.tcpReader != null) {
            this.tcpReader.printStats();
        }
        this.icmpDecoder.printStats();
    }

    public Packet decode(byte[] packetData, int ipStart, long packetTimestampSecs, long packetTimestampMicros, boolean partial) {
        ++this.counter;
        if (ipStart == -1) {
            return Packet.NULL;
        }
        Packet packet = this.createPacket(packetData, ipStart);
        if (packet == Packet.NULL) {
            return packet;
        }
        packet.setTsMilli(packetTimestampSecs * 1000L + (long)Math.round((float)packetTimestampMicros / 1000.0f));
        packet.setData(packetData);
        packet.setIpStart(ipStart);
        int ipProtocolHeaderVersion = IPv4Util.getInternetProtocolHeaderVersion(packetData, ipStart);
        packet.setIpVersion(ipProtocolHeaderVersion);
        int totalLength = 0;
        if (ipProtocolHeaderVersion == 4) {
            int ipHeaderLen = IPv4Util.getInternetProtocolHeaderLength(packetData, ipStart);
            packet.setIpHeaderLen(ipHeaderLen);
            packet.setTtl(IPv4Util.decodeTTL(packetData, ipStart));
            packet.setSrcAddr(IPv4Util.decodeSrc(packetData, ipStart));
            packet.setDstAddr(IPv4Util.decodeDst(packetData, ipStart));
            if (packet.getSrcAddr() != null) {
                packet.setSrc(packet.getSrcAddr().getHostAddress());
            }
            if (packet.getDstAddr() != null) {
                packet.setDst(packet.getDstAddr().getHostAddress());
            }
            packet.setIpId(IPv4Util.decodeId(packetData, ipStart));
            totalLength = PcapReaderUtil.convertShort(packetData, ipStart + 2);
            this.decodeV4Fragmented(packet, ipStart, packetData);
        } else {
            int ipHeaderLen = IPv6Util.getInternetProtocolHeaderLength(packetData, ipStart);
            packet.setIpHeaderLen(ipHeaderLen);
            packet.setTtl(IPv6Util.decodeTTL(packetData, ipStart));
            packet.setSrcAddr(IPv6Util.decodeSrc(packetData, ipStart));
            packet.setDstAddr(IPv6Util.decodeDst(packetData, ipStart));
            if (packet.getSrcAddr() != null) {
                packet.setSrc(packet.getSrcAddr().getHostAddress());
            }
            if (packet.getDstAddr() != null) {
                packet.setDst(packet.getDstAddr().getHostAddress());
            }
            packet.setIpId(IPv6Util.decodeId(packetData, ipStart));
            int payloadLength = PcapReaderUtil.convertShort(packetData, ipStart + 4);
            totalLength = payloadLength + 40;
            this.decodeV6Fragmented(packet, ipStart, packetData);
            if (packet.isFragmented()) {
                IPv6Util.buildInternetProtocolV6ExtensionHeaderFragment(packet, packetData, ipStart);
            }
        }
        packet.setTotalLength(totalLength);
        if (partial) {
            return packet;
        }
        this.lastPacketTs = packet.getTsMilli();
        return this.decode(packet, packetData, ipStart);
    }

    public Packet decode(Packet packet) {
        ++this.counterPayload;
        if (packet == Packet.NULL || packet == Packet.LAST) {
            return packet;
        }
        Packet p = this.decode(packet, packet.getData(), packet.getIpStart());
        p.setData(null);
        return p;
    }

    public Packet decode(Packet packet, byte[] packetData, int ipStart) {
        int padding = Math.max(packetData.length - (packet.getTotalLength() + ipStart), 0);
        byte[] reassembledData = this.reassemble(packet, packetData = Arrays.copyOfRange(packetData, ipStart + packet.getIpHeaderLen(), packetData.length - padding));
        if (reassembledData.length == 0) {
            return Packet.NULL;
        }
        return this.handlePayload(packet, reassembledData);
    }

    public Packet createPacket(byte[] packetData, int ipStart) {
        int ipProtocolHeaderVersion = IPv4Util.getInternetProtocolHeaderVersion(packetData, ipStart);
        byte protocol = -1;
        if (ipProtocolHeaderVersion == 4) {
            protocol = IPv4Util.decodeProtocol(packetData, ipStart);
        } else if (ipProtocolHeaderVersion == 6) {
            protocol = IPv6Util.decodeProtocol(packetData, ipStart);
        } else {
            log.error("Unsupported IP version " + ipProtocolHeaderVersion + " ipstart=" + ipStart);
            return Packet.NULL;
        }
        return PacketFactory.create(protocol);
    }

    private Packet handlePayload(Packet packet, byte[] packetData) {
        if (1 == packet.getProtocol() || 58 == packet.getProtocol()) {
            this.icmpDecoder.reassemble((ICMPPacket)packet, packetData);
            return packet;
        }
        if (6 == packet.getProtocol()) {
            if (this.tcpReader == null) {
                return Packet.NULL;
            }
            packet = this.tcpReader.reassemble(packet, packetData);
        } else if (17 == packet.getProtocol()) {
            packet = this.udpReader.reassemble(packet, packetData);
        }
        if (packet instanceof DNSPacket && ((DNSPacket)packet).getMessageCount() == 0) {
            return Packet.NULL;
        }
        return packet;
    }

    private void decodeV6Fragmented(Packet packet, int ipStart, byte[] packetData) {
        byte nxtHdr = packetData[ipStart + 6];
        packet.setFragmented(nxtHdr == 44);
    }

    private void decodeV4Fragmented(Packet packet, int ipStart, byte[] packetData) {
        long fragmentOffset = (long)(PcapReaderUtil.convertShort(packetData, ipStart + 6) & 0x1FFF) * 8L;
        packet.setFragOffset(fragmentOffset);
        int flags = packetData[ipStart + 6] & 0xE0;
        if ((flags & 0x40) == 64) {
            packet.setDoNotFragment(true);
        } else if ((flags & 0x20) == 32 || fragmentOffset != 0L) {
            packet.setFragmented(true);
            packet.setLastFragment((flags & 0x20) == 0 && fragmentOffset != 0L);
        }
    }

    public byte[] reassemble(Packet packet, byte[] packetData) {
        if (!packet.isFragmented()) {
            return packetData;
        }
        Datagram datagram = packet.getDatagram();
        DatagramPayload payload = new DatagramPayload(packet.getFragOffset(), packetData);
        this.datagrams.put((Object)datagram, (Object)payload);
        if (packet.isLastFragment()) {
            byte[] reassembledPacketData = new byte[]{};
            Collection datagramPayloads = this.datagrams.removeAll((Object)datagram);
            if (datagramPayloads != null && !datagramPayloads.isEmpty()) {
                int reassembledFragments = 0;
                DatagramPayload prev = null;
                for (DatagramPayload datagramPayload : datagramPayloads) {
                    if (prev == null && datagramPayload.getOffset() != 0L) {
                        if (log.isDebugEnabled()) {
                            log.debug("Datagram chain not starting at 0. Probably received packets out-of-order. Can't reassemble this packet.");
                        }
                        return new byte[0];
                    }
                    if (prev != null && !datagramPayload.linked(prev)) {
                        if (log.isDebugEnabled()) {
                            log.debug("Broken datagram chain between " + String.valueOf(datagramPayload) + " and " + String.valueOf(prev) + ". Can't reassemble this packet.");
                        }
                        return new byte[0];
                    }
                    reassembledPacketData = Bytes.concat((byte[][])new byte[][]{reassembledPacketData, datagramPayload.getPayload()});
                    ++reassembledFragments;
                    prev = datagramPayload;
                }
                packet.setReassembledFragments(reassembledFragments);
            }
            return reassembledPacketData;
        }
        return new byte[0];
    }

    public Multimap<Datagram, DatagramPayload> getDatagrams() {
        return this.datagrams;
    }

    public void setDatagrams(Multimap<Datagram, DatagramPayload> datagrams) {
        this.datagrams = datagrams;
    }

    public void setTcpFlows(Map<TCPFlow, FlowData> flows) {
        if (this.tcpReader != null && this.tcpReader instanceof TCPDecoder) {
            ((TCPDecoder)this.tcpReader).setFlows(flows);
        }
    }

    public void clearCache(int ipFragmentTTL) {
        long max = this.lastPacketTs - (long)ipFragmentTTL;
        List dgExpiredList = this.datagrams.keySet().stream().filter(k -> k.getTime() < max).collect(Collectors.toList());
        log.info("------------- IP Decoder Cache Stats ---------------------");
        log.info("IP datagram cache size: " + this.datagrams.size());
        log.info("Expired (to be removed) IP datagrams: " + dgExpiredList.size());
        dgExpiredList.stream().forEach(dg -> this.datagrams.removeAll(dg));
    }

    public void reset() {
        this.udpReader.reset();
        if (this.tcpReader != null) {
            this.tcpReader.reset();
        }
        this.icmpDecoder.reset();
    }

    public int getCounter() {
        return this.counter;
    }

    public int getCounterPayload() {
        return this.counterPayload;
    }

    public Decoder getTcpReader() {
        return this.tcpReader;
    }

    public Decoder getUdpReader() {
        return this.udpReader;
    }

    public ICMPDecoder getIcmpDecoder() {
        return this.icmpDecoder;
    }

    public long getLastPacketTs() {
        return this.lastPacketTs;
    }

    public void setCounter(int counter) {
        this.counter = counter;
    }

    public void setCounterPayload(int counterPayload) {
        this.counterPayload = counterPayload;
    }

    public void setTcpReader(Decoder tcpReader) {
        this.tcpReader = tcpReader;
    }

    public void setUdpReader(Decoder udpReader) {
        this.udpReader = udpReader;
    }

    public void setIcmpDecoder(ICMPDecoder icmpDecoder) {
        this.icmpDecoder = icmpDecoder;
    }

    public void setLastPacketTs(long lastPacketTs) {
        this.lastPacketTs = lastPacketTs;
    }
}

