/*
 * Decompiled with CFR 0.152.
 */
package nl.stokpop.eventscheduler;

import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import nl.stokpop.eventscheduler.EventBroadcaster;
import nl.stokpop.eventscheduler.api.CustomEvent;
import nl.stokpop.eventscheduler.api.EventLogger;
import nl.stokpop.eventscheduler.api.EventProperties;
import nl.stokpop.eventscheduler.api.EventSchedulerSettings;
import nl.stokpop.eventscheduler.api.SchedulerExceptionHandler;
import nl.stokpop.eventscheduler.api.SchedulerExceptionType;
import nl.stokpop.eventscheduler.api.TestContext;
import nl.stokpop.eventscheduler.exception.EventSchedulerRuntimeException;
import nl.stokpop.eventscheduler.exception.handler.SchedulerHandlerException;

class EventSchedulerEngine {
    private final EventLogger logger;
    private ScheduledExecutorService executorKeepAlive;
    private ScheduledExecutorService executorCustomEvents;

    EventSchedulerEngine(EventLogger logger) {
        if (logger == null) {
            throw new EventSchedulerRuntimeException("logger is null");
        }
        this.logger = logger;
    }

    void startKeepAliveThread(TestContext context, EventSchedulerSettings settings, EventBroadcaster broadcaster, EventProperties eventProperties, SchedulerExceptionHandler schedulerExceptionHandler) {
        this.nullChecks(context, broadcaster, eventProperties);
        if (this.executorKeepAlive != null) {
            throw new RuntimeException("cannot start keep alive thread multiple times!");
        }
        this.logger.info(String.format("calling keep alive every %s", settings.getKeepAliveDuration()));
        this.executorKeepAlive = this.createKeepAliveScheduler();
        KeepAliveRunner keepAliveRunner = new KeepAliveRunner(context, broadcaster, eventProperties, schedulerExceptionHandler);
        this.executorKeepAlive.scheduleAtFixedRate(keepAliveRunner, 0L, settings.getKeepAliveDuration().getSeconds(), TimeUnit.SECONDS);
    }

    private void nullChecks(TestContext context, EventBroadcaster broadcaster, EventProperties eventProperties) {
        if (context == null) {
            throw new NullPointerException("TestContext cannot be null");
        }
        if (broadcaster == null) {
            throw new NullPointerException("EventBroadcaster cannot be null");
        }
        if (eventProperties == null) {
            throw new NullPointerException("EventProperties cannot be null");
        }
    }

    private void addToExecutor(ScheduledExecutorService executorService, CustomEvent event, EventBroadcaster broadcaster) {
        executorService.schedule(new EventRunner(event, broadcaster), event.getDuration().getSeconds(), TimeUnit.SECONDS);
    }

    void shutdownThreadsNow() {
        List<Runnable> runnables;
        this.logger.info("shutdown Executor threads");
        if (this.executorKeepAlive != null) {
            this.executorKeepAlive.shutdownNow();
        }
        if (this.executorCustomEvents != null && (runnables = this.executorCustomEvents.shutdownNow()).size() > 0) {
            if (runnables.size() == 1) {
                this.logger.warn("There is 1 custom event that is not (fully) executed!");
            } else {
                this.logger.warn("There are " + runnables.size() + " custom events that are not (fully) executed!");
            }
        }
        this.executorKeepAlive = null;
        this.executorCustomEvents = null;
    }

    void startCustomEventScheduler(TestContext context, List<CustomEvent> scheduleEvents, EventBroadcaster broadcaster, EventProperties eventProperties) {
        this.nullChecks(context, broadcaster, eventProperties);
        if (scheduleEvents != null && !scheduleEvents.isEmpty()) {
            this.logger.info(EventSchedulerEngine.createEventScheduleMessage(scheduleEvents));
            this.executorCustomEvents = this.createCustomEventScheduler();
            scheduleEvents.forEach(event -> this.addToExecutor(this.executorCustomEvents, (CustomEvent)event, broadcaster));
        } else {
            this.logger.info("no custom schedule events found");
        }
    }

    public static String createEventScheduleMessage(List<CustomEvent> scheduleEvents) {
        StringBuilder message = new StringBuilder();
        message.append("=== custom events schedule ===");
        scheduleEvents.forEach(event -> message.append("\n==> ").append(String.format("ScheduleEvent %-36.36s [fire-at=%-8s settings=%-50.50s]", event.getNameDescription(), event.getDuration(), event.getSettings())));
        return message.toString();
    }

    private ScheduledExecutorService createKeepAliveScheduler() {
        return Executors.newSingleThreadScheduledExecutor(r -> {
            String threadName = "Keep-Alive-Thread";
            this.logger.info("create new thread: " + threadName);
            return new Thread(r, threadName);
        });
    }

    private ScheduledExecutorService createCustomEventScheduler() {
        return Executors.newScheduledThreadPool(2, new ThreadFactory(){
            private final AtomicInteger threadCount = new AtomicInteger(0);

            @Override
            public Thread newThread(Runnable r) {
                String threadName = "Custom-Event-Thread-" + this.threadCount.incrementAndGet();
                EventSchedulerEngine.this.logger.info("create new thread: " + threadName);
                return new Thread(r, threadName);
            }
        });
    }

    class EventRunner
    implements Runnable {
        private final CustomEvent event;
        private final EventBroadcaster eventBroadcaster;

        public EventRunner(CustomEvent event, EventBroadcaster eventBroadcaster) {
            this.event = event;
            this.eventBroadcaster = eventBroadcaster;
        }

        @Override
        public void run() {
            try {
                this.eventBroadcaster.broadcastCustomEvent(this.event);
            }
            catch (Exception e) {
                EventSchedulerEngine.this.logger.error("Broadcast custom event failed", e);
            }
        }

        public String toString() {
            return String.format("EventRunner for event %s", this.event);
        }
    }

    class KeepAliveRunner
    implements Runnable {
        private final TestContext context;
        private final EventBroadcaster broadcaster;
        private final EventProperties eventProperties;
        private final SchedulerExceptionHandler schedulerExceptionHandler;

        KeepAliveRunner(TestContext context, EventBroadcaster broadcaster, EventProperties eventProperties, SchedulerExceptionHandler schedulerExceptionHandler) {
            this.context = context;
            this.broadcaster = broadcaster;
            this.eventProperties = eventProperties;
            this.schedulerExceptionHandler = schedulerExceptionHandler;
        }

        @Override
        public void run() {
            try {
                this.broadcaster.broadcastKeepAlive();
            }
            catch (SchedulerHandlerException e) {
                this.handleException(e);
            }
            catch (Exception e) {
                EventSchedulerEngine.this.logger.error("Broadcast keep-alive failed", e);
            }
        }

        private void handleException(SchedulerHandlerException e) {
            String message = e.getMessage();
            SchedulerExceptionType exceptionType = e.getExceptionType();
            if (this.schedulerExceptionHandler != null) {
                EventSchedulerEngine.this.logger.info("SchedulerHandlerException found, invoke " + (Object)((Object)exceptionType) + " on SchedulerExceptionHandler: " + message);
                if (exceptionType == SchedulerExceptionType.KILL) {
                    this.schedulerExceptionHandler.kill(e.getMessage());
                } else if (exceptionType == SchedulerExceptionType.ABORT) {
                    this.schedulerExceptionHandler.abort(e.getMessage());
                }
            } else {
                EventSchedulerEngine.this.logger.warn("SchedulerHandlerException " + (Object)((Object)exceptionType) + " was thrown, but no SchedulerExceptionHandler is present. Message: " + message);
            }
        }

        public String toString() {
            return "KeepAliveRunner for " + this.context.getTestRunId();
        }
    }
}

