/*
 * Decompiled with CFR 0.152.
 */
package nl.stokpop.eventscheduler;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import nl.stokpop.eventscheduler.EventBroadcaster;
import nl.stokpop.eventscheduler.api.CustomEvent;
import nl.stokpop.eventscheduler.api.EventLogger;
import nl.stokpop.eventscheduler.api.SchedulerExceptionHandler;
import nl.stokpop.eventscheduler.api.SchedulerExceptionType;
import nl.stokpop.eventscheduler.exception.EventSchedulerRuntimeException;
import nl.stokpop.eventscheduler.exception.handler.SchedulerHandlerException;

class EventSchedulerEngine {
    private final EventLogger logger;
    private ScheduledExecutorService executorKeepAlive;
    private ScheduledExecutorService executorCustomEvents;

    EventSchedulerEngine(EventLogger logger) {
        if (logger == null) {
            throw new EventSchedulerRuntimeException("logger is null");
        }
        this.logger = logger;
    }

    void startKeepAliveThread(String name, Duration keepAliveDuration, EventBroadcaster broadcaster, SchedulerExceptionHandler schedulerExceptionHandler) {
        this.nullChecks(name, broadcaster);
        if (this.executorKeepAlive != null) {
            throw new RuntimeException("cannot start keep alive thread multiple times!");
        }
        this.logger.info(String.format("calling keep alive every %s", keepAliveDuration));
        this.executorKeepAlive = this.createKeepAliveScheduler();
        KeepAliveRunner keepAliveRunner = new KeepAliveRunner(name, broadcaster, schedulerExceptionHandler);
        this.executorKeepAlive.scheduleAtFixedRate(keepAliveRunner, 0L, keepAliveDuration.getSeconds(), TimeUnit.SECONDS);
    }

    private void nullChecks(EventBroadcaster broadcaster) {
        if (broadcaster == null) {
            throw new NullPointerException("eventBroadcaster cannot be null");
        }
    }

    private void nullChecks(String name, EventBroadcaster broadcaster) {
        if (name == null) {
            throw new NullPointerException("name cannot be null");
        }
        this.nullChecks(broadcaster);
    }

    private void addToExecutor(ScheduledExecutorService executorService, CustomEvent event, EventBroadcaster broadcaster) {
        executorService.schedule(new EventRunner(event, broadcaster), event.getDuration().getSeconds(), TimeUnit.SECONDS);
    }

    void shutdownThreadsNow() {
        List<Runnable> runnables;
        this.logger.info("shutdown Executor threads");
        if (this.executorKeepAlive != null) {
            this.executorKeepAlive.shutdownNow();
        }
        if (this.executorCustomEvents != null && (runnables = this.executorCustomEvents.shutdownNow()).size() > 0) {
            if (runnables.size() == 1) {
                this.logger.warn("There is 1 custom event that is not (fully) executed!");
            } else {
                this.logger.warn("There are " + runnables.size() + " custom events that are not (fully) executed!");
            }
        }
        this.executorKeepAlive = null;
        this.executorCustomEvents = null;
    }

    void startCustomEventScheduler(Collection<CustomEvent> scheduleEvents, EventBroadcaster broadcaster) {
        this.nullChecks(broadcaster);
        if (scheduleEvents != null && !scheduleEvents.isEmpty()) {
            this.logger.info(EventSchedulerEngine.createEventScheduleMessage(scheduleEvents));
            this.executorCustomEvents = this.createCustomEventScheduler();
            scheduleEvents.forEach(event -> this.addToExecutor(this.executorCustomEvents, (CustomEvent)event, broadcaster));
        } else {
            this.logger.info("no custom schedule events found");
        }
    }

    public static String createEventScheduleMessage(Collection<CustomEvent> scheduleEvents) {
        StringBuilder message = new StringBuilder();
        message.append("=== custom events schedule ===");
        scheduleEvents.forEach(event -> message.append("\n==> ").append(String.format("ScheduleEvent %-36.36s [fire-at=%-8s settings=%-50.50s]", event.getNameDescription(), event.getDuration(), event.getSettings())));
        return message.toString();
    }

    private ScheduledExecutorService createKeepAliveScheduler() {
        return Executors.newSingleThreadScheduledExecutor(r -> {
            String threadName = "Keep-Alive-Thread";
            this.logger.info("create new thread: " + threadName);
            return new Thread(r, threadName);
        });
    }

    private ScheduledExecutorService createCustomEventScheduler() {
        return Executors.newScheduledThreadPool(2, new ThreadFactory(){
            private final AtomicInteger threadCount = new AtomicInteger(0);

            @Override
            public Thread newThread(Runnable r) {
                String threadName = "Custom-Event-Thread-" + this.threadCount.incrementAndGet();
                EventSchedulerEngine.this.logger.info("create new thread: " + threadName);
                return new Thread(r, threadName);
            }
        });
    }

    class EventRunner
    implements Runnable {
        private final CustomEvent event;
        private final EventBroadcaster eventBroadcaster;

        public EventRunner(CustomEvent event, EventBroadcaster eventBroadcaster) {
            this.event = event;
            this.eventBroadcaster = eventBroadcaster;
        }

        @Override
        public void run() {
            try {
                this.eventBroadcaster.broadcastCustomEvent(this.event);
            }
            catch (Exception e) {
                EventSchedulerEngine.this.logger.error("Broadcast custom event failed", e);
            }
        }

        public String toString() {
            return String.format("EventRunner for event %s", this.event);
        }
    }

    class KeepAliveRunner
    implements Runnable {
        private final String name;
        private final EventBroadcaster broadcaster;
        private final SchedulerExceptionHandler schedulerExceptionHandler;

        KeepAliveRunner(String name, EventBroadcaster broadcaster, SchedulerExceptionHandler schedulerExceptionHandler) {
            this.name = name;
            this.broadcaster = broadcaster;
            this.schedulerExceptionHandler = schedulerExceptionHandler;
        }

        @Override
        public void run() {
            try {
                this.broadcaster.broadcastKeepAlive();
            }
            catch (SchedulerHandlerException e) {
                this.handleException(e);
            }
            catch (Exception e) {
                EventSchedulerEngine.this.logger.error("Broadcast keep-alive failed", e);
            }
        }

        private void handleException(SchedulerHandlerException e) {
            String message = e.getMessage();
            SchedulerExceptionType exceptionType = e.getExceptionType();
            if (this.schedulerExceptionHandler != null) {
                EventSchedulerEngine.this.logger.info("SchedulerHandlerException found, invoke " + (Object)((Object)exceptionType) + " on SchedulerExceptionHandler: " + message);
                if (exceptionType == SchedulerExceptionType.KILL) {
                    this.schedulerExceptionHandler.kill(e.getMessage());
                } else if (exceptionType == SchedulerExceptionType.ABORT) {
                    this.schedulerExceptionHandler.abort(e.getMessage());
                }
            } else {
                EventSchedulerEngine.this.logger.warn("SchedulerHandlerException " + (Object)((Object)exceptionType) + " was thrown, but no SchedulerExceptionHandler is present. Message: " + message);
            }
        }

        public String toString() {
            return "KeepAliveRunner: " + this.name;
        }
    }
}

