/*-
 * #%L
 * Mule CoAP Connector
 * %%
 * Copyright (C) 2019 - 2024 (teslanet.nl) Rogier Cobben
 * 
 * Contributors:
 *     (teslanet.nl) Rogier Cobben - initial creation
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package nl.teslanet.mule.connectors.coap.api.options;


import java.util.List;

import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

import nl.teslanet.mule.connectors.coap.api.query.QueryParam;


/**
 * The option parameters of a CoAP response.
 *
 */
public class ResponseOptionsParams
{
    /**
     * RFC 7252: The Content-Format Option indicates the representation format of
     * the message payload. The representation format is given as a numeric
     * Content-Format identifier that is defined in the "CoAP Content-Formats"
     * registry (Section 12.3).
     * 
     * @see <a href=
     *      "https://tools.ietf.org/html/rfc7252#section-5.10.3">IETF RFC 7252 -
     *      5.10.3. Content-Format</a>
     */
    @Parameter
    @Optional
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary( "The Content-Format Option indicates the representation format of the message payload." )
    @Placement( tab= "Options", order= 50 )
    private Integer contentFormat= null;

    /**
     * RFC 7252: The Max-Age Option indicates the maximum time a response may be
     * cached before it is considered not fresh (see Section 5.6.1).
     * 
     * The option value is an integer number of seconds between 0 and
     * {@code 2**32-1 } inclusive (about 136.1 years). A default value of 60 seconds
     * is assumed in the absence of the option in a response.
     * 
     * @see <a href=
     *      "https://tools.ietf.org/html/rfc7252#section-5.10.5">IETF RFC 7252 -
     *      5.10.5. Max-Age</a>
     */
    @Parameter
    @Optional
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary(
        "The Max-Age Option indicates the maximum time a response may be cached\n before it is considered not fresh, in [Seconds]."
    )
    @Placement( tab= "Options", order= 51 )
    private Long maxAge= null;

    /**
     * RFC 7252: An entity-tag is intended for use as a resource-local identifier
     * for differentiating between representations of the same resource that vary
     * over time. It is generated by the server providing the resource, which may
     * generate it in any number of ways including a version, checksum, hash, or
     * time. An endpoint receiving an entity-tag MUST treat it as opaque and make no
     * assumptions about its content or structure.
     * 
     * The ETag Option in a response provides the current value (i.e., after the
     * request was processed) of the entity-tag for the "tagged representation".
     * 
     * In a GET request, an endpoint that has one or more representations previously
     * obtained from the resource, and has obtained ETag response options with
     * these, can specify an instance of the ETag Option for one or more of these
     * stored responses. A server can issue a 2.03 Valid response (Section 5.9.1.3)
     * in place of a 2.05 Content response if one of the ETags given is the
     * entity-tag for the current representation, i.e., is valid; the 2.03 Valid
     * response then echoes this specific ETag in a response option.
     * 
     * @see <a href=
     *      "https://tools.ietf.org/html/rfc7252#section-5.10.6">IETF RFC 7252 -
     *      5.10.6. ETag</a>
     */
    @Parameter
    @Optional
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary(
        "The entity-tag of the response content is a resource-local identifier for differentiating between representations of the same resource."
    )
    @Placement( tab= "Options", order= 52 )
    private TypedValue< Object > entityTagValue= null;

    /**
     * RFC 7252: The Location-Path and Location-Query Options together indicate a
     * relative URI that consists either of an absolute path, a query string, or
     * both. A combination of these options is included in a 2.01 (Created) response
     * to indicate the location of the resource created as the result of a POST
     * request (see Section 5.8.2). The location is resolved relative to the request
     * URI.
     * 
     * Location-Path Options specify the absolute path to the resource.
     * 
     * @see <a href=
     *      "https://tools.ietf.org/html/rfc7252#section-5.10.7">IETF RFC 7252 -
     *      5.10.7. Location-Path and Location-Query</a>
     */
    @Parameter
    @Optional
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary( "The Location-Path specifies the absolute path to a resource that is created." )
    @Example( "/some/location/on/server" )
    @Placement( tab= "Options", order= 53 )
    private String locationPath= null;

    /**
     * RFC 7252: The Location-Path and Location-Query Options together indicate a
     * relative URI that consists either of an absolute path, a query string, or
     * both. A combination of these options is included in a 2.01 (Created) response
     * to indicate the location of the resource created as the result of a POST
     * request (see Section 5.8.2). The location is resolved relative to the request
     * URI.
     * 
     * Each each Location-Query Option specifies one argument parameterizing the
     * resource.
     * 
     * @see <a href=
     *      "https://tools.ietf.org/html/rfc7252#section-5.10.7">IETF RFC 7252 -
     *      5.10.7. Location-Path and Location-Query</a>
     */
    @Parameter
    @Optional
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary( "The Location-Query specifies the arguments parameterizing the resource that is created." )
    @Placement( tab= "Options", order= 54 )
    private List< QueryParam > locationQuery= null;

    /**
     * RFC 7959: In a response carrying a Block2 Option, to indicate the current
     *  estimate the server has of the total size of the resource
     * representation, measured in bytes ("size indication").
     * 
     * @see <a href=
     *      "https://datatracker.ietf.org/doc/html/rfc7959#section-4">IETF RFC 7959 - 4. The Size2 and Size1 Options</a>
     */
    @Parameter
    @Optional
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary( "Indication of the response payload size in [Bytes]." )
    @Placement( tab= "Options", order= 55 )
    private Integer responseSize= null;

    /**
     * RFC 7959: The Size1 Option may be used: In a 4.13 response, to indicate the maximum size that would have
     * been acceptable, measured in bytes.
     * 
     * @see <a href=
     *      "https://datatracker.ietf.org/doc/html/rfc7959#section-4">IETF RFC 7959 - 4. The Size2 and Size1 Options</a>
     */
    @Parameter
    @Optional
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary( "To indicate (in a 4.13 response) the maximum payload size that would have been acceptable, in [Bytes]." )
    @Placement( tab= "Options", order= 56 )
    private Integer acceptableRequestSize= null;

    /**
     * RFC 8613: The OSCORE option indicates that the CoAP message is an OSCORE
     * message and that it contains a compressed COSE object.
     * 
     * The OSCORE option includes the OSCORE flag bits.
     * 
     * @see <a href=
     *      "https://tools.ietf.org/html/rfc8613#section-2">IETF RFC 8613 - 2. The
     *      OSCORE Option</a>
     */
    //TODO add oscore feature
    //    @Parameter
    //    @Optional
    //    @Expression(ExpressionSupport.SUPPORTED)
    //    @Summary("The OSCORE option includes the OSCORE flag bits.")
    //    private String oscore= null;

    /**
     * The other CoAP options to send with the response.
     */
    @Parameter
    @Optional
    @NullSafe
    @Expression( ExpressionSupport.SUPPORTED )
    @Summary( "The other CoAP options to send with the response." )
    @Placement( tab= "Options", order= 59 )
    private List< OtherOption > otherOptions;

    /**
     * @return the otherResponseOptions
     */
    public List< OtherOption > getOtherOptions()
    {
        return otherOptions;
    }

    /**
     * @param otherOptions the other options to set
     */
    public void setOtherOptions( List< OtherOption > otherOptions )
    {
        this.otherOptions= otherOptions;
    }

    /**
     * @return the contentFormat
     */
    public Integer getContentFormat()
    {
        return contentFormat;
    }

    /**
     * @param contentFormat the contentFormat to set
     */
    public void setContentFormat( Integer contentFormat )
    {
        this.contentFormat= contentFormat;
    }

    /**
     * @return the maxAge
     */
    public Long getMaxAge()
    {
        return maxAge;
    }

    /**
     * @param maxAge the maxAge to set
     */
    public void setMaxAge( Long maxAge )
    {
        this.maxAge= maxAge;
    }

    /**
     * @return The Entity-tag value of the resource.
     */
    public TypedValue< Object > getEntityTagValue()
    {
        return entityTagValue;
    }

    /**
     * @param entityTagValue The Entity-tag value of the resource to set
     */
    public void setEntityTagValue( TypedValue< Object > entityTagValue )
    {
        this.entityTagValue= entityTagValue;
    }

    /**
     * @return the locationPath
     */
    public String getLocationPath()
    {
        return locationPath;
    }

    /**
     * @param locationPath the locationPath to set
     */
    public void setLocationPath( String locationPath )
    {
        this.locationPath= locationPath;
    }

    /**
     * @return the locationQuery
     */
    public List< QueryParam > getLocationQuery()
    {
        return locationQuery;
    }

    /**
     * @param locationQuery the locationQuery to set
     */
    public void setLocationQuery( List< QueryParam > locationQuery )
    {
        this.locationQuery= locationQuery;
    }

    /**
     * @return the responseSize
     */
    public Integer getResponseSize()
    {
        return responseSize;
    }

    /**
     * @param responseSize the responseSize to set
     */
    public void setResponseSize( Integer responseSize )
    {
        this.responseSize= responseSize;
    }

    /**
     * @return the acceptableRequestSize
     */
    public Integer getAcceptableRequestSize()
    {
        return acceptableRequestSize;
    }

    /**
     * @param acceptableRequestSize the acceptableRequestSize to set
     */
    public void setAcceptableRequestSize( Integer acceptableRequestSize )
    {
        this.acceptableRequestSize= acceptableRequestSize;
    }
}
