/*
 * Decompiled with CFR 0.152.
 */
package no.digipost.security.cert;

import java.io.IOException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.time.Clock;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import no.digipost.security.DigipostSecurity;
import no.digipost.security.DigipostSecurityException;
import no.digipost.security.cert.CertStatus;
import no.digipost.security.cert.CertificateValidatorConfig;
import no.digipost.security.cert.OcspDecision;
import no.digipost.security.cert.OcspPolicy;
import no.digipost.security.cert.ReviewedCertPath;
import no.digipost.security.cert.RevocationReason;
import no.digipost.security.cert.Trust;
import no.digipost.security.cert.TrustedCertificateAndIssuer;
import no.digipost.security.ocsp.OcspLookup;
import no.digipost.security.ocsp.OcspResult;
import no.digipost.security.ocsp.OcspUtils;
import org.apache.http.impl.client.CloseableHttpClient;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.CertificateStatus;
import org.bouncycastle.cert.ocsp.OCSPException;
import org.bouncycastle.cert.ocsp.RevokedStatus;
import org.bouncycastle.cert.ocsp.SingleResp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateValidator {
    private static final Logger LOG = LoggerFactory.getLogger(CertificateValidator.class);
    private final CertificateValidatorConfig config;
    private final CloseableHttpClient client;
    private final Map<X509Certificate, ResultWithTime> cache = new HashMap<X509Certificate, ResultWithTime>();
    private final Trust trust;
    private final Clock clock;

    public CertificateValidator(Trust trust, CloseableHttpClient httpClient) {
        this(CertificateValidatorConfig.MOST_STRICT, trust, httpClient);
    }

    public CertificateValidator(CertificateValidatorConfig config, Trust trust, CloseableHttpClient httpClient) {
        this.config = config;
        this.trust = trust;
        this.client = httpClient;
        this.clock = trust.clock;
    }

    public CertStatus validateCert(Certificate certificate) {
        return this.validateCert(certificate, this.config);
    }

    private CertStatus validateCert(Certificate certificate, CertificateValidatorConfig config) {
        ReviewedCertPath certPath;
        ResultWithTime cachedResult;
        if (!(certificate instanceof X509Certificate)) {
            LOG.warn("Tried to validate a non-" + X509Certificate.class.getSimpleName() + ": " + certificate.getType() + "(" + certificate.getClass().getName() + ")");
            return CertStatus.UNTRUSTED;
        }
        X509Certificate x509Certificate = DigipostSecurity.requireX509(certificate);
        if (this.cache.containsKey(x509Certificate)) {
            cachedResult = this.cache.get(x509Certificate);
            if (!cachedResult.isExpiredAt(this.clock.instant()) || cachedResult.status == CertStatus.REVOKED) {
                return cachedResult.status;
            }
        } else {
            cachedResult = null;
        }
        if (!(certPath = this.trust.resolveCertPath(x509Certificate)).isTrusted()) {
            return CertStatus.UNTRUSTED;
        }
        TrustedCertificateAndIssuer trustedCertificateAndIssuer = certPath.getTrustedCertificateAndIssuer();
        if (config.ocspPolicy.decideFor(trustedCertificateAndIssuer) == OcspDecision.LOOKUP_OCSP) {
            CertStatus ocspStatus = this.ocspLookup(trustedCertificateAndIssuer, config);
            if (ocspStatus != CertStatus.OK && config.allowsOcspResult(ocspStatus)) {
                LOG.info("Status {} for certificate {} is configured as {}", new Object[]{ocspStatus, DigipostSecurity.describe(certificate), CertStatus.OK});
                ocspStatus = CertStatus.OK;
            }
            if (ocspStatus == CertStatus.UNDECIDED && cachedResult != null && cachedResult.isValidAsStaleValueForFailedResolvingOfNewValue(this.clock.instant())) {
                if (cachedResult.shouldLogIfUsedAsStaleValue(this.clock.instant())) {
                    LOG.error("OCSP older than 2 hours: {}", (Object)DigipostSecurity.describe(x509Certificate));
                }
                cachedResult.setUnexpiredFrom(this.clock.instant());
                return cachedResult.status;
            }
            this.cache.put(x509Certificate, new ResultWithTime(this.clock.instant(), ocspStatus));
            return ocspStatus;
        }
        this.cache.put(x509Certificate, new ResultWithTime(this.clock.instant(), CertStatus.OK));
        return CertStatus.OK;
    }

    private CertStatus ocspLookup(TrustedCertificateAndIssuer certificateAndIssuer, CertificateValidatorConfig config) {
        return certificateAndIssuer.ocspLookupRequest.map(OcspLookup::new).flatMap(lookup -> {
            try {
                return Optional.of(lookup.executeUsing(this.client));
            }
            catch (RuntimeException e) {
                LOG.warn("Failed {} for {}: {} '{}'", new Object[]{lookup, certificateAndIssuer, e.getClass().getSimpleName(), e.getMessage()});
                if (LOG.isDebugEnabled()) {
                    LOG.debug(e.getClass().getSimpleName() + ": '" + e.getMessage() + "'", (Throwable)e);
                }
                return Optional.empty();
            }
        }).map(result -> {
            try (OcspResult ocspResult = result;){
                X509Certificate ocspSignatureValidationCertificate;
                BasicOCSPResp basix;
                if (!ocspResult.isOkResponse()) {
                    LOG.warn("Unexpected OCSP response ({}) for {}.", (Object)ocspResult.response.getStatusLine(), (Object)certificateAndIssuer);
                    CertStatus certStatus = CertStatus.UNDECIDED;
                    return certStatus;
                }
                try {
                    basix = ocspResult.getResponseObject();
                }
                catch (IllegalStateException | CertIOException | OCSPException e) {
                    LOG.warn("OCSP response for {}, error reading the response because: {} '{}'", new Object[]{certificateAndIssuer, e.getClass().getSimpleName(), e.getMessage()});
                    CertStatus certStatus = CertStatus.UNDECIDED;
                    if (ocspResult == null) return certStatus;
                    ocspResult.close();
                    return certStatus;
                }
                if (basix == null) {
                    LOG.warn("OCSP response for {}, returned a null response, this could be a problem with the certificate issuer", (Object)certificateAndIssuer);
                    CertStatus e = CertStatus.UNDECIDED;
                    return e;
                }
                Optional<X509Certificate> ocspSigningCertificate = CertificateValidator.findOcspSigningCertificate(basix, config);
                if (ocspSigningCertificate.isPresent()) {
                    ocspSignatureValidationCertificate = ocspSigningCertificate.get();
                    CertStatus certStatus = this.validateCert(ocspSignatureValidationCertificate, config.withOcspPolicy(OcspPolicy.NEVER_DO_OCSP_LOOKUP));
                    if (certStatus != CertStatus.OK) {
                        LOG.warn("OCSP signing certificate is '{}': {}", (Object)certStatus, (Object)DigipostSecurity.describe(ocspSignatureValidationCertificate));
                        CertStatus certStatus2 = certStatus;
                        return certStatus2;
                    }
                } else {
                    ocspSignatureValidationCertificate = certificateAndIssuer.issuer;
                }
                if (!config.ocspSignatureValidator.isValidSignature(basix, ocspSignatureValidationCertificate)) {
                    LOG.warn("OCSP response for {} returnerte et svar som feilet signaturvalidering", (Object)certificateAndIssuer);
                    CertStatus certStatus = CertStatus.UNDECIDED;
                    return certStatus;
                }
                for (SingleResp cresp : basix.getResponses()) {
                    if (cresp.getCertStatus() == CertificateStatus.GOOD) continue;
                    if (cresp.getCertStatus() instanceof RevokedStatus) {
                        RevokedStatus s = (RevokedStatus)cresp.getCertStatus();
                        RevocationReason reason = Optional.of(s).filter(RevokedStatus::hasRevocationReason).map(r -> RevocationReason.resolve(r.getRevocationReason())).orElse(RevocationReason.unspecified);
                        LOG.warn("OCSP response for {} returned status revoked: {}, reason: '{}'", new Object[]{certificateAndIssuer, s.getRevocationTime(), reason});
                        CertStatus certStatus = CertStatus.REVOKED;
                        return certStatus;
                    }
                    LOG.warn("OCSP response for {} returned status {}", (Object)certificateAndIssuer, (Object)cresp.getCertStatus().getClass().getSimpleName());
                    CertStatus certStatus = CertStatus.UNDECIDED;
                    return certStatus;
                }
                LOG.debug("OCSP response for {} returned status GOOD", (Object)certificateAndIssuer);
                CertStatus certStatus = CertStatus.OK;
                return certStatus;
            }
            catch (IOException | OCSPException e) {
                throw new DigipostSecurityException(e);
            }
        }).orElse(CertStatus.UNDECIDED);
    }

    private static Optional<X509Certificate> findOcspSigningCertificate(BasicOCSPResp basix, CertificateValidatorConfig config) {
        Optional<X509Certificate> ocspSigningCertificate;
        if (config.ignoreCustomSigningCertificatesInOcspResponses) {
            return Optional.empty();
        }
        try {
            ocspSigningCertificate = OcspUtils.findOscpSigningCertificate(basix);
        }
        catch (Exception e) {
            LOG.warn("Unexpected error while loooking for OCSP signing certificate in OCSP-response. {}: '{}'", new Object[]{e.getClass().getSimpleName(), e.getMessage(), e});
            ocspSigningCertificate = Optional.empty();
        }
        return ocspSigningCertificate;
    }

    private static class ResultWithTime {
        public final CertStatus status;
        private final Instant reallyExpires;
        private final Instant warnAfter;
        private Instant expires;

        public ResultWithTime(Instant validated, CertStatus status) {
            this.status = status;
            if (status == CertStatus.UNDECIDED) {
                this.reallyExpires = this.expires = validated.plus(1L, ChronoUnit.MINUTES);
                this.warnAfter = this.expires;
            } else {
                this.expires = validated.plus(5L, ChronoUnit.MINUTES);
                this.reallyExpires = validated.plus(48L, ChronoUnit.HOURS);
                this.warnAfter = validated.plus(2L, ChronoUnit.HOURS);
            }
        }

        void setUnexpiredFrom(Instant instant) {
            this.expires = instant.plus(1L, ChronoUnit.MINUTES);
        }

        boolean isExpiredAt(Instant instant) {
            return !instant.isBefore(this.expires);
        }

        boolean isValidAsStaleValueForFailedResolvingOfNewValue(Instant instant) {
            return instant.isBefore(this.reallyExpires);
        }

        boolean shouldLogIfUsedAsStaleValue(Instant instant) {
            return instant.isAfter(this.warnAfter);
        }
    }
}

