/*
 * Decompiled with CFR 0.152.
 */
package no.digipost.security.cert;

import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.CertPath;
import java.security.cert.CertPathBuilder;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.CertificateFactory;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.time.Clock;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.security.auth.x500.X500Principal;
import no.digipost.security.DigipostSecurity;
import no.digipost.security.DigipostSecurityException;
import no.digipost.security.cert.CertHelper;
import no.digipost.security.cert.MissingTrustAnchorException;
import no.digipost.security.cert.NonMatchingClocksException;
import no.digipost.security.cert.ReviewedCertPath;
import no.digipost.security.keystore.KeyStoreType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Trust {
    private static final Logger LOG = LoggerFactory.getLogger(Trust.class);
    private final Map<X500Principal, Set<X509Certificate>> trustAnchorCerts;
    private final Map<X500Principal, Set<X509Certificate>> trustedIntermediateCerts;
    final Clock clock;

    public static Trust in(Clock clock, X509Certificate ... trustedCertificates) {
        return Trust.in(clock, Stream.of(trustedCertificates));
    }

    public static Trust in(Clock clock, Stream<X509Certificate> trustedCertificates) {
        Map grouped = trustedCertificates.collect(Collectors.groupingBy(TrustBasis::determineFrom, Collectors.toSet()));
        return new Trust(grouped.getOrDefault((Object)TrustBasis.ANCHOR, Collections.emptySet()).stream(), grouped.getOrDefault((Object)TrustBasis.DERIVED, Collections.emptySet()).stream(), clock);
    }

    public static Trust merge(Trust t1, Trust t2) {
        if (!Objects.equals(t1.clock, t2.clock)) {
            throw new NonMatchingClocksException(t1.clock, t2.clock);
        }
        return new Trust(Trust.mergeMultimaps(t1.trustAnchorCerts, t2.trustAnchorCerts), Trust.mergeMultimaps(t1.trustedIntermediateCerts, t2.trustedIntermediateCerts), t1.clock);
    }

    private static <K, V> Map<K, Set<V>> mergeMultimaps(Map<K, Set<V>> m1, Map<K, Set<V>> m2) {
        HashMap merged = new HashMap();
        for (Map.Entry<K, Set<V>> m1Entry : m1.entrySet()) {
            K m1key = m1Entry.getKey();
            HashSet union = new HashSet(m1Entry.getValue());
            union.addAll(m2.getOrDefault(m1key, Collections.emptySet()));
            merged.put(m1key, Collections.unmodifiableSet(union));
        }
        for (Map.Entry<K, Set<V>> m2Entry : m2.entrySet()) {
            K m2Key = m2Entry.getKey();
            if (merged.containsKey(m2Key)) continue;
            merged.put(m2Key, m2Entry.getValue());
        }
        return Collections.unmodifiableMap(merged);
    }

    public Trust(Stream<X509Certificate> trustAnchorCertificates, Stream<X509Certificate> intermediateCertificates) {
        this(trustAnchorCertificates, intermediateCertificates, Clock.systemDefaultZone());
    }

    public Trust(Stream<X509Certificate> trustAnchorCertificates, Stream<X509Certificate> intermediateCertificates, Clock clock) {
        this(Collections.unmodifiableMap(trustAnchorCertificates.collect(Collectors.groupingBy(X509Certificate::getSubjectX500Principal, Collectors.toSet()))), Collections.unmodifiableMap(intermediateCertificates.collect(Collectors.groupingBy(X509Certificate::getSubjectX500Principal, Collectors.toSet()))), clock);
    }

    private Trust(Map<X500Principal, Set<X509Certificate>> trustAnchorCerts, Map<X500Principal, Set<X509Certificate>> trustedIntermediateCerts, Clock clock) {
        this.trustAnchorCerts = Objects.requireNonNull(trustAnchorCerts, "trust anchor certificates");
        this.trustedIntermediateCerts = Objects.requireNonNull(trustedIntermediateCerts, "intermediate certificates");
        this.clock = Objects.requireNonNull(clock, "clock");
        this.validate();
    }

    private void validate() {
        List<X509Certificate> intermediateCertsWithoutAnchor = this.trustedIntermediateCerts.values().stream().flatMap(Collection::stream).filter(cert -> !this.trustAnchorCerts.containsKey(cert.getIssuerX500Principal())).collect(Collectors.toList());
        if (!intermediateCertsWithoutAnchor.isEmpty()) {
            throw new MissingTrustAnchorException(intermediateCertsWithoutAnchor);
        }
    }

    public ReviewedCertPath resolveCertPath(X509Certificate certificate) {
        try {
            CollectionCertStoreParameters certStoreParams = new CollectionCertStoreParameters(this.getTrustAnchorsAndAnyIntermediateCertificatesFor(certificate.getIssuerX500Principal()).collect(Collectors.toSet()));
            X509CertSelector certSelector = new X509CertSelector();
            certSelector.setCertificate(certificate);
            certSelector.setSubject(certificate.getSubjectX500Principal());
            certSelector.setCertificateValid(Date.from(this.clock.instant()));
            CertStore certStore = CertStore.getInstance("Collection", certStoreParams);
            PKIXBuilderParameters params = new PKIXBuilderParameters(this.getTrustAnchors(), (CertSelector)certSelector);
            params.addCertStore(certStore);
            params.setSigProvider("BC");
            params.setRevocationEnabled(false);
            params.setDate(Date.from(this.clock.instant()));
            CertPath certpath = CertPathBuilder.getInstance("PKIX").build(params).getCertPath();
            if (certpath.getCertificates().size() > 1) {
                return new ReviewedCertPath(certpath, this::trusts);
            }
            CertificateFactory cf = DigipostSecurity.getX509CertificateFactory();
            Optional<X509Certificate> issuer = CertHelper.findTrustAnchorCert(certificate, this.getTrustAnchors());
            return new ReviewedCertPath(cf.generateCertPath(Stream.concat(Stream.of(certificate), issuer.map(Stream::of).orElse(Stream.empty())).collect(Collectors.toList())), path -> issuer.isPresent());
        }
        catch (GeneralSecurityException e) {
            LOG.warn("Error generating cert path for certificate, because the issuer is not trusted. {}: {}. certificate: {}", new Object[]{e.getClass().getSimpleName(), e.getMessage(), DigipostSecurity.describe(certificate)});
            if (LOG.isDebugEnabled()) {
                LOG.debug(e.getClass().getSimpleName() + ": '" + e.getMessage() + "'", (Throwable)e);
            }
            return new ReviewedCertPath(e);
        }
    }

    public boolean trusts(CertPath certPath) {
        try {
            Set<TrustAnchor> trustAnchors = this.getTrustAnchors();
            PKIXParameters params = new PKIXParameters(trustAnchors);
            params.setSigProvider("BC");
            params.setRevocationEnabled(false);
            params.setDate(Date.from(this.clock.instant()));
            CertPathValidator.getInstance("PKIX").validate(certPath, params);
            return true;
        }
        catch (CertPathValidatorException e) {
            return false;
        }
        catch (GeneralSecurityException e) {
            throw new DigipostSecurityException(e);
        }
    }

    public Set<TrustAnchor> getTrustAnchors() {
        return this.getTrustAnchorCertificates().stream().map(c -> new TrustAnchor((X509Certificate)c, null)).collect(Collectors.toSet());
    }

    public Set<X509Certificate> getTrustAnchorCertificates() {
        return this.trustAnchorCerts.values().stream().flatMap(Collection::stream).collect(Collectors.toSet());
    }

    public KeyStore getTrustAnchorsKeyStore() {
        return KeyStoreType.JCEKS.newKeyStore().containing(this.getTrustAnchorCertificates()).withNoPassword();
    }

    public KeyStore asKeyStore() {
        Stream trustAnchors = this.getTrustAnchorCertificates().stream();
        Stream intermediateCerts = this.trustedIntermediateCerts.values().stream().flatMap(certs -> certs.stream());
        return KeyStoreType.JCEKS.newKeyStore().containing(Stream.concat(trustAnchors, intermediateCerts)).withNoPassword();
    }

    public Map<X500Principal, Set<X509Certificate>> getTrustedIntermediateCertificates() {
        return this.trustedIntermediateCerts;
    }

    Stream<X509Certificate> getTrustAnchorsAndAnyIntermediateCertificatesFor(X500Principal principal) {
        return Stream.concat(this.getTrustAnchorCertificates().stream(), this.getTrustedIntermediateCertificates().getOrDefault(principal, Collections.emptySet()).stream());
    }

    public boolean equals(Object other) {
        if (other instanceof Trust) {
            Trust that = (Trust)other;
            return Objects.equals(this.clock, that.clock) && Objects.equals(this.trustAnchorCerts, that.trustAnchorCerts) && Objects.equals(this.trustedIntermediateCerts, that.trustedIntermediateCerts);
        }
        return false;
    }

    public int hashCode() {
        return Objects.hash(this.clock, this.trustAnchorCerts, this.trustedIntermediateCerts);
    }

    private static enum TrustBasis {
        ANCHOR,
        DERIVED;


        static TrustBasis determineFrom(X509Certificate cert) {
            return cert.getSubjectX500Principal().equals(cert.getIssuerX500Principal()) ? ANCHOR : DERIVED;
        }
    }
}

