/*
 * Decompiled with CFR 0.152.
 */
package dev.cel.common.internal;

import com.google.common.base.Strings;
import com.google.common.math.LongMath;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.google.protobuf.Duration;
import com.google.protobuf.Timestamp;
import dev.cel.common.annotations.Internal;
import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

@Internal
public final class ProtoTimeUtils {
    private static final long TIMESTAMP_SECONDS_MIN = -62135596800L;
    private static final long TIMESTAMP_SECONDS_MAX = 253402300799L;
    private static final long DURATION_SECONDS_MIN = -315576000000L;
    private static final long DURATION_SECONDS_MAX = 315576000000L;
    private static final int MILLIS_PER_SECOND = 1000;
    private static final int NANOS_PER_SECOND = 1000000000;
    private static final int NANOS_PER_MILLISECOND = 1000000;
    private static final int NANOS_PER_MICROSECOND = 1000;
    private static final long SECONDS_PER_MINUTE = 60L;
    private static final long SECONDS_PER_HOUR = 3600L;
    private static final ThreadLocal<SimpleDateFormat> TIMESTAMP_FORMAT = new ThreadLocal<SimpleDateFormat>(){

        @Override
        protected SimpleDateFormat initialValue() {
            return ProtoTimeUtils.createTimestampFormat();
        }
    };
    public static final Timestamp TIMESTAMP_EPOCH = Timestamp.newBuilder().setSeconds(0L).setNanos(0).build();
    public static final Duration DURATION_ZERO = Duration.newBuilder().setSeconds(0L).setNanos(0).build();

    private static SimpleDateFormat createTimestampFormat() {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.ENGLISH);
        GregorianCalendar calendar = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
        calendar.setGregorianChange(new Date(Long.MIN_VALUE));
        sdf.setCalendar(calendar);
        return sdf;
    }

    public static Timestamp toProtoTimestamp(Instant instant) {
        return ProtoTimeUtils.normalizedTimestamp(instant.getEpochSecond(), instant.getNano());
    }

    public static Duration toProtoDuration(java.time.Duration duration) {
        return ProtoTimeUtils.normalizedDuration(duration.getSeconds(), duration.getNano());
    }

    public static Instant toJavaInstant(Timestamp timestamp) {
        timestamp = ProtoTimeUtils.normalizedTimestamp(timestamp.getSeconds(), timestamp.getNanos());
        return Instant.ofEpochSecond(timestamp.getSeconds(), timestamp.getNanos());
    }

    public static java.time.Duration toJavaDuration(Duration duration) {
        duration = ProtoTimeUtils.normalizedDuration(duration.getSeconds(), duration.getNanos());
        return java.time.Duration.ofSeconds(duration.getSeconds(), duration.getNanos());
    }

    public static long toSeconds(Timestamp timestamp) {
        return ProtoTimeUtils.checkValid(timestamp).getSeconds();
    }

    public static long toSeconds(Duration duration) {
        return ProtoTimeUtils.checkValid(duration).getSeconds();
    }

    public static long toHours(Duration duration) {
        return ProtoTimeUtils.checkValid(duration).getSeconds() / 3600L;
    }

    public static long toMinutes(Duration duration) {
        return ProtoTimeUtils.checkValid(duration).getSeconds() / 60L;
    }

    public static long toMillis(Duration duration) {
        ProtoTimeUtils.checkValid(duration);
        return LongMath.checkedAdd(LongMath.checkedMultiply(duration.getSeconds(), 1000L), duration.getNanos() / 1000000);
    }

    public static Timestamp fromSecondsToTimestamp(long seconds) {
        return ProtoTimeUtils.normalizedTimestamp(seconds, 0);
    }

    public static Duration fromSecondsToDuration(long seconds) {
        return ProtoTimeUtils.normalizedDuration(seconds, 0);
    }

    public static Duration fromMillisToDuration(long milliseconds) {
        return ProtoTimeUtils.normalizedDuration(milliseconds / 1000L, (int)(milliseconds % 1000L * 1000000L));
    }

    @CanIgnoreReturnValue
    private static Duration checkValid(Duration duration) {
        int nanos;
        long seconds = duration.getSeconds();
        if (!ProtoTimeUtils.isDurationValid(seconds, nanos = duration.getNanos())) {
            throw new IllegalArgumentException(Strings.lenientFormat("Duration is not valid. See proto definition for valid values. Seconds (%s) must be in range [-315,576,000,000, +315,576,000,000]. Nanos (%s) must be in range [-999,999,999, +999,999,999]. Nanos must have the same sign as seconds", seconds, nanos));
        }
        return duration;
    }

    @CanIgnoreReturnValue
    private static Timestamp checkValid(Timestamp timestamp) {
        int nanos;
        long seconds = timestamp.getSeconds();
        if (!ProtoTimeUtils.isTimestampValid(seconds, nanos = timestamp.getNanos())) {
            throw new IllegalArgumentException(Strings.lenientFormat("Timestamp is not valid. See proto definition for valid values. Seconds (%s) must be in range [-62,135,596,800, +253,402,300,799]. Nanos (%s) must be in range [0, +999,999,999].", seconds, nanos));
        }
        return timestamp;
    }

    public static String toString(Timestamp timestamp) {
        ProtoTimeUtils.checkValid(timestamp);
        long seconds = timestamp.getSeconds();
        int nanos = timestamp.getNanos();
        StringBuilder result = new StringBuilder();
        Date date = new Date(seconds * 1000L);
        result.append(TIMESTAMP_FORMAT.get().format(date));
        if (nanos != 0) {
            result.append(".");
            result.append(ProtoTimeUtils.formatNanos(nanos));
        }
        result.append("Z");
        return result.toString();
    }

    public static String toString(Duration duration) {
        ProtoTimeUtils.checkValid(duration);
        long seconds = duration.getSeconds();
        int nanos = duration.getNanos();
        StringBuilder result = new StringBuilder();
        if (seconds < 0L || nanos < 0) {
            result.append("-");
            seconds = -seconds;
            nanos = -nanos;
        }
        result.append(seconds);
        if (nanos != 0) {
            result.append(".");
            result.append(ProtoTimeUtils.formatNanos(nanos));
        }
        result.append("s");
        return result.toString();
    }

    public static Timestamp parse(String value) throws ParseException {
        int nanos;
        String timeValue;
        int dayOffset = value.indexOf(84);
        if (dayOffset == -1) {
            throw new ParseException("Failed to parse timestamp: invalid timestamp \"" + value + "\"", 0);
        }
        int timezoneOffsetPosition = value.indexOf(90, dayOffset);
        if (timezoneOffsetPosition == -1) {
            timezoneOffsetPosition = value.indexOf(43, dayOffset);
        }
        if (timezoneOffsetPosition == -1) {
            timezoneOffsetPosition = value.indexOf(45, dayOffset);
        }
        if (timezoneOffsetPosition == -1) {
            throw new ParseException("Failed to parse timestamp: missing valid timezone offset.", 0);
        }
        String secondValue = timeValue = value.substring(0, timezoneOffsetPosition);
        String nanoValue = "";
        int pointPosition = timeValue.indexOf(46);
        if (pointPosition != -1) {
            secondValue = timeValue.substring(0, pointPosition);
            nanoValue = timeValue.substring(pointPosition + 1);
        }
        Date date = TIMESTAMP_FORMAT.get().parse(secondValue);
        long seconds = date.getTime() / 1000L;
        int n = nanos = nanoValue.isEmpty() ? 0 : ProtoTimeUtils.parseNanos(nanoValue);
        if (value.charAt(timezoneOffsetPosition) == 'Z') {
            if (value.length() != timezoneOffsetPosition + 1) {
                throw new ParseException("Failed to parse timestamp: invalid trailing data \"" + value.substring(timezoneOffsetPosition) + "\"", 0);
            }
        } else {
            String offsetValue = value.substring(timezoneOffsetPosition + 1);
            long offset = ProtoTimeUtils.parseTimezoneOffset(offsetValue);
            seconds = value.charAt(timezoneOffsetPosition) == '+' ? (seconds -= offset) : (seconds += offset);
        }
        try {
            return ProtoTimeUtils.normalizedTimestamp(seconds, nanos);
        }
        catch (IllegalArgumentException e) {
            ParseException ex = new ParseException("Failed to parse timestamp " + value + " Timestamp is out of range.", 0);
            ex.initCause(e);
            throw ex;
        }
    }

    public static Duration add(Duration d1, Duration d2) {
        java.time.Duration javaDuration1 = ProtoTimeUtils.toJavaDuration(ProtoTimeUtils.checkValid(d1));
        java.time.Duration javaDuration2 = ProtoTimeUtils.toJavaDuration(ProtoTimeUtils.checkValid(d2));
        java.time.Duration sum = javaDuration1.plus(javaDuration2);
        return ProtoTimeUtils.toProtoDuration(sum);
    }

    public static Timestamp add(Timestamp ts, Duration dur) {
        Instant javaInstant = ProtoTimeUtils.toJavaInstant(ProtoTimeUtils.checkValid(ts));
        java.time.Duration javaDuration = ProtoTimeUtils.toJavaDuration(ProtoTimeUtils.checkValid(dur));
        Instant newInstant = javaInstant.plus(javaDuration);
        return ProtoTimeUtils.toProtoTimestamp(newInstant);
    }

    public static Duration subtract(Duration d1, Duration d2) {
        java.time.Duration javaDuration1 = ProtoTimeUtils.toJavaDuration(ProtoTimeUtils.checkValid(d1));
        java.time.Duration javaDuration2 = ProtoTimeUtils.toJavaDuration(ProtoTimeUtils.checkValid(d2));
        java.time.Duration sum = javaDuration1.minus(javaDuration2);
        return ProtoTimeUtils.toProtoDuration(sum);
    }

    public static Timestamp subtract(Timestamp ts, Duration dur) {
        Instant javaInstant = ProtoTimeUtils.toJavaInstant(ProtoTimeUtils.checkValid(ts));
        java.time.Duration javaDuration = ProtoTimeUtils.toJavaDuration(ProtoTimeUtils.checkValid(dur));
        Instant newInstant = javaInstant.minus(javaDuration);
        return ProtoTimeUtils.toProtoTimestamp(newInstant);
    }

    public static Duration between(Timestamp from, Timestamp to) {
        Instant javaFrom = ProtoTimeUtils.toJavaInstant(ProtoTimeUtils.checkValid(from));
        Instant javaTo = ProtoTimeUtils.toJavaInstant(ProtoTimeUtils.checkValid(to));
        java.time.Duration between = java.time.Duration.between(javaFrom, javaTo);
        return ProtoTimeUtils.toProtoDuration(between);
    }

    public static int compare(Duration x, Duration y) {
        return DurationComparator.INSTANCE.compare(x, y);
    }

    public static int compare(Timestamp x, Timestamp y) {
        return TimestampComparator.INSTANCE.compare(x, y);
    }

    public static Timestamp now() {
        Instant nowInstant = Instant.now();
        return Timestamp.newBuilder().setSeconds(nowInstant.getEpochSecond()).setNanos(nowInstant.getNano()).build();
    }

    private static long parseTimezoneOffset(String value) throws ParseException {
        int pos = value.indexOf(58);
        if (pos == -1) {
            throw new ParseException("Invalid offset value: " + value, 0);
        }
        String hours = value.substring(0, pos);
        String minutes = value.substring(pos + 1);
        try {
            return (Long.parseLong(hours) * 60L + Long.parseLong(minutes)) * 60L;
        }
        catch (NumberFormatException e) {
            ParseException ex = new ParseException("Invalid offset value: " + value, 0);
            ex.initCause(e);
            throw ex;
        }
    }

    private static int parseNanos(String value) throws ParseException {
        int result = 0;
        for (int i = 0; i < 9; ++i) {
            result *= 10;
            if (i >= value.length()) continue;
            if (value.charAt(i) < '0' || value.charAt(i) > '9') {
                throw new ParseException("Invalid nanoseconds.", 0);
            }
            result += value.charAt(i) - 48;
        }
        return result;
    }

    private static boolean isDurationValid(long seconds, int nanos) {
        if (seconds < -315576000000L || seconds > 315576000000L) {
            return false;
        }
        if ((long)nanos < -999999999L || nanos >= 1000000000) {
            return false;
        }
        return seconds >= 0L && nanos >= 0 || seconds <= 0L && nanos <= 0;
    }

    private static boolean isTimestampValid(long seconds, int nanos) {
        if (!ProtoTimeUtils.isTimestampSecondsValid(seconds)) {
            return false;
        }
        return nanos >= 0 && nanos < 1000000000;
    }

    private static boolean isTimestampSecondsValid(long seconds) {
        return seconds >= -62135596800L && seconds <= 253402300799L;
    }

    private static Timestamp normalizedTimestamp(long seconds, int nanos) {
        if (nanos <= -1000000000 || nanos >= 1000000000) {
            seconds = LongMath.checkedAdd(seconds, nanos / 1000000000);
            nanos %= 1000000000;
        }
        if (nanos < 0) {
            nanos += 1000000000;
            seconds = LongMath.checkedSubtract(seconds, 1L);
        }
        Timestamp timestamp = Timestamp.newBuilder().setSeconds(seconds).setNanos(nanos).build();
        return ProtoTimeUtils.checkValid(timestamp);
    }

    private static Duration normalizedDuration(long seconds, int nanos) {
        if (nanos <= -1000000000 || nanos >= 1000000000) {
            seconds = LongMath.checkedAdd(seconds, nanos / 1000000000);
            nanos %= 1000000000;
        }
        if (seconds > 0L && nanos < 0) {
            nanos += 1000000000;
            --seconds;
        }
        if (seconds < 0L && nanos > 0) {
            nanos -= 1000000000;
            ++seconds;
        }
        Duration duration = Duration.newBuilder().setSeconds(seconds).setNanos(nanos).build();
        return ProtoTimeUtils.checkValid(duration);
    }

    private static String formatNanos(int nanos) {
        if (nanos % 1000000 == 0) {
            return String.format(Locale.ENGLISH, "%1$03d", nanos / 1000000);
        }
        if (nanos % 1000 == 0) {
            return String.format(Locale.ENGLISH, "%1$06d", nanos / 1000);
        }
        return String.format(Locale.ENGLISH, "%1$09d", nanos);
    }

    private ProtoTimeUtils() {
    }

    private static enum DurationComparator implements Comparator<Duration>,
    Serializable
    {
        INSTANCE;


        @Override
        public int compare(Duration d1, Duration d2) {
            ProtoTimeUtils.checkValid(d1);
            ProtoTimeUtils.checkValid(d2);
            int secDiff = Long.compare(d1.getSeconds(), d2.getSeconds());
            return secDiff != 0 ? secDiff : Integer.compare(d1.getNanos(), d2.getNanos());
        }
    }

    private static enum TimestampComparator implements Comparator<Timestamp>,
    Serializable
    {
        INSTANCE;


        @Override
        public int compare(Timestamp t1, Timestamp t2) {
            ProtoTimeUtils.checkValid(t1);
            ProtoTimeUtils.checkValid(t2);
            int secDiff = Long.compare(t1.getSeconds(), t2.getSeconds());
            return secDiff != 0 ? secDiff : Integer.compare(t1.getNanos(), t2.getNanos());
        }
    }
}

