/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

/**
 * A Message represents all the data in a message.
 */
@SuppressWarnings("nls")
public class Message implements CLogLevels, Serializable {

	private Severity severity;

	private List<Message> messages = new LinkedList<Message>();
    /**
     * Initial size of @{link #description}.
     */
    private static final int TEXT_SIZE = 2;

    /** Generator for runtime IDs. */
    private static Random generator = new Random();

    /** Runtime identifier (to correlate logs). */
    private final String runtimeId;

    /** Message number (ID). */
    private String messageID;

    /** Message arguments */
    private Object[] args;

    /** Kind of message (<code>MSGTYPE_xxx</code>). */
    private MessageType messageType = MessageTypeEnum.NONE;

    /** Bit flag of log levels. */
    private long logLevel = LOG_DEFAULT;

    /** Message title. */
    private String title;

    /** Main message text. Never NULL. */
    private String messageText = "";

    /** Text for tool tip / status bar etc. */
    private String hint;

    /** The additional descriptive texts of this message. */
    private List<String>description = new ArrayList<String> (TEXT_SIZE);

    /** Valid replies for this message (<code>REPLSET_xxx</code>). */
    private ReplySetType validReplies = ReplySetType.REPLSET_NONE;

    /** Actual reply (<code>REPLY_xxx</code>) when input is not a string. */
    private MessageReplyType reply = MessageReplyType.REPLY_NONE;

    /**
     * Actual reply (<code>String</code>) when input is a string. Also set to
     * default value as necessary.
     */
    private String replString = "";

    /** Holds the replied file if the message involves a file chooser.*/
    private File replyFile= null;

    /** Keep exception. */
    private Throwable thrown;

    /** The context for the message */
    private MessageContext context;

    /** Default constructor for Message. */
    public Message() {
    	this(null, (Object) null);
    }

    /**
     * Constructor with message id.
     *
     * @param messageID Message identificator
     */
    public Message(final String messageID) {
        this.messageID = messageID;
        runtimeId = String.valueOf(generator.nextLong());
    }

    /**
     * Constructor with message id and arguments.
     * 
     * @param messageID Message identificator
     * @param args message arguments
     */
    public Message(String messageID, Object... args) {
        this.messageID = messageID;
        this.args = args;
        runtimeId = String.valueOf(generator.nextLong());
    }


    /**
     * Constructor with message number and type.
     *
     * @param messageID Message id
     * @param args message arguments
     * @param messageType Message type (<code>MSGTYPE_xxx</code>-constant)
     */
    public Message(String messageID, MessageType messageType, Object... args) {
        this(messageID,args);
        this.messageType = messageType;
    }

    /**
     * Returns a String representation of this object.
     *
     * @return This object as a String.
     */
    @SuppressWarnings("nls")
	@Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("Message[").append(this.runtimeId)
               .append(" id ").append(this.messageID)
               .append(" type ").append(this.messageType)
               .append(" text ").append(this.messageText)
               .append("]");
        if (this.reply != null && !MessageReplyType.REPLY_NONE.equals(this.reply)) {
            sb.append("=").append(this.reply);
        }
        else if (this.replString != null && !"".equals(this.replString)) {
            sb.append("=").append(this.replString);
        }
        else if (this.replyFile!=null) {
            sb.append("=").append(this.replyFile);
        }
        return sb.toString().trim();
    }

    /**
     * Returns the runtime ID (message identifier).
     *
     * @return The runtime ID.
     */
    public String getRuntimeId() {
        return runtimeId;
    }

    /**
     * Sets the message number.
     *
     * @param messageID New value of MessageID.
     * @see #getMessageID()
     */
    public void setMessageID(String messageID) {
        this.messageID = messageID;
    }

    /**
     * Returns the message number.
     *
     * @return Message number.
     * @see #setMessageID(String)
     */
    public String getMessageID() {
        return this.messageID;
    }

    /**
     * Sets the message arguments.
     *
     * @param pArgs New value of args.
     * @see #getArgs()
     */
    public void setArgs(Object[] pArgs) {
        this.args = pArgs.clone();
    }

    /**
     * Returns the message arguments.
     *
     * @return Message arguments.
     * @see #setArgs(Object[])
     */
    public Object[] getArgs() {
        return (this.args != null) ? this.args.clone() : null;
    }

    /**
     * Sets valid replies for this message.
     *
     * @param pReplies A REPLSET_xxx-constant for reply set.
     * @see #getValidReplies()
     */
    public void setValidReplies(final ReplySetType pReplies) {
        this.validReplies = pReplies;
    }

    /**
     * Returns a REPLSET_xxx-constant for the valid replies to this message.
     *
     * @return REPLSET_xxx-constant.
     * @see #setValidReplies(ReplySetType)
     */
    public ReplySetType getValidReplies() {
        return this.validReplies;
    }

    /**
     * Sets the actual reply.
     *
     * @param pRepl Selected reply as a REPLY_xxx-constant.
     * @see #getReply()
     */
    public void setReply(final MessageReplyType pRepl) {
        this.reply = pRepl;
    }

    /**
     * Returns the selected reply as a REPLY_xxx-constant.
     *
     * @return The reply as a REPLY_xxx-constant.
     * @see #setReply(MessageReplyType)
     */
    public MessageReplyType getReply() {
        return this.reply;
    }

    /**
     * Sets the message type.
     *
     * @param pMsgType Message type as a MSGTYPE_xxx-constant..
     * @see #getMsgType()
     */
    public void setMsgType(final MessageType pMsgType) {
        this.messageType = pMsgType;
    }

    /**
     * Returns the message type as a MSGTYPE_xxx-constant.
     *
     * @return The message type as a MSGTYPE_xxx-constant.
     * @see #setMsgType(MessageType)
     */
    public MessageType getMsgType() {
         return this.messageType;
    }

    /**
     * Sets new log level.
     *
     * @param pLogLevel New log level as a LOG_xxx-constant.
     * @see #getLogLevel()
     */
    public void setLogLevel(final long pLogLevel) {
        this.logLevel = pLogLevel;
    }

    /**
     * Returns the log level as a LOG_xxx-constant.
     *
     * @return The log level as a LOG_xxx-constant.
     * @see #setLogLevel(long)
     */
    public long getLogLevel() {
        return this.logLevel;
    }

    /**
     * Sets new value of one of the texts.
     *
     * @param index Index of the text to set. Must be &gt;=0. If the index
     *            creates a "hole" in the array, empty strings are added in
     *            between.
     * @param txt The text to store.
     * @throws IndexOutOfBoundsException If index is negative
     * @see #getDescription(int)
     * @see #setDescription(int, String)
     */
    public void setDescription(final int index, final MessageText txt) {
        setDescription(index, txt.toString());
    }

    /**
     * Sets new value of one of the texts - to a fixed String.
     *
     * @param index Index of the text to set. Must be &gt;=0. If the index
     *            creates a "hole" in the array, empty strings are added in
     *            between.
     * @param txt The text to store.
     * @throws IndexOutOfBoundsException If index is negative
     * @see #getDescription(int)
     * @see #setDescription(int, MessageText)
     * @see #addDescription(String)
     */
    public void setDescription(final int index, final String txt) {
        if (index < 0) {
            throw new IndexOutOfBoundsException();
        }
        final int currSize = this.description.size();
        if (index >= currSize) {
            for (int i = currSize; i <= index; i++) {
                this.description.add(null);
            }
        }
        this.description.set(index, txt);
    }

    /**
     * Adds another descriptive string.
     *
     * @param txt The text to add.
     * @see #getDescription(int)
     * @see #setDescription(int, String)
     */
    public void addDescription(final String txt) {
        this.description.add(txt);
    }

    /**
     * Clears all texts.
     *
     * @see #setDescription(int, MessageText)
     */
    public void clearDescription() {
        this.description.clear();
    }

    /**
     * Returns one of the description texts.
     *
     * @param index Index of the text to get. Must be &gt;=0. Returns an empty
     *            text if this index is not set.
     * @return The text.
     * @throws IndexOutOfBoundsException If index is negative
     * @see #setDescription(int, MessageText)
     */
    @SuppressWarnings("nls")
	public String getDescription(final int index) {
        if (index < 0) {
            throw new IndexOutOfBoundsException();
        }
        String mtx = null;
        if (index < this.description.size()) {
            mtx = this.description.get(index);
        }
        return (mtx == null) ? "" : mtx;
    }

    /**
     * Returns all the description texts.
     *
     * @param delimiter Inserted beween all elements. May be null.
     * @return The text.
     * @see #getDescription(int)
     */
    public String getDescription(final String delimiter) {
        // (farjam) When this part got upgraded to java 5, it was done in a way
        // that ensured same functionality as the previos version regarding
        // delimiters (we still get delimiters between nulls)
        final StringBuffer sb = new StringBuffer();
        boolean addDelimiter = false;
        for (String s : description) {
            if (addDelimiter) {
                if (delimiter != null) sb.append(delimiter);
            } else {
                addDelimiter = true;
            }
            if (s != null) {
                sb.append(s);
            }
        }
        return sb.toString();
    }

    /**
     * Adds information from another message into this.
     *
     * @param that Message to copy from
     */
    public void addMessage(final Message that) {
        if (that == null)
            return;
        messages.add(that);
        
    }


    /**
     * Copies information from another message into this message. Most fields
     * are copied only if the original field is empty. Exceptions: logLevel is
     * combined (bitwise), and all description fields are added.
     *
     * @param that Message to copy from
     */
    public void copy(final Message that) {
        if (that == null)
            return;
        if (this.messageID == null) {
            this.messageID= that.messageID;
        }

        this.logLevel |= that.logLevel;

        if (this.thrown == null) {
            this.thrown= that.thrown;
        }
        if (that.description != null) {
            this.description.addAll(that.description);
        }
        if (this.messageType == null ||
            this.messageType.equals(MessageTypeEnum.NONE)) {
            this.messageType = that.messageType;
        }
        if (this.reply == null ||
            this.reply.equals(MessageReplyType.REPLY_NONE)) {
            this.reply = that.reply;
        }
        if (this.validReplies == null ||
            this.validReplies.equals(ReplySetType.REPLSET_NONE)) {
            this.validReplies = that.validReplies;
        }
        if (Message.isEmpty(this.messageText)) {
            this.messageText = that.messageText;
        }
        if (Message.isEmpty(this.hint)) {
            this.hint = that.hint;
        }
        if (Message.isEmpty(this.title)) {
            this.title = that.title;
        }
        if (Message.isEmpty(this.replString)) {
            this.replString = that.replString;
        }
    }

    /**
     * Convenience method, checks if the specified String is either empty (""),
     * or consists entirely of blanks (" "), or is a null-pointer, in which case
     * it returns <code>true</code>
     *
     * @param s the string to check
     * @return Returns <code>true</code> if the string is either empty or
     *         <code>null</code>
     */
    public static boolean isEmpty(String s) {
        return s == null || s.trim().length() == 0;
    }


    /**
     * Set new value of title.
     *
     * @param pTitle New value of title.
     * @see #getTitle()
     */
    public void setTitle(String pTitle) {
        this.title = pTitle;
    }

    /**
     * Returns the title.
     *
     * @return The title.
     * @see #setTitle(String)
     */
    public String getTitle() {
        return this.title;
    }

    /**
     * Set new main message text.
     *
     * @param pMessageText New main text.
     * @see #getMessageText()
     */
    @SuppressWarnings("nls")
	public void setMessageText(final String pMessageText) {
        this.messageText = (pMessageText == null) ? "" : pMessageText;
    }

    /**
     * Returns the main message text. Never null.
     *
     * @return The main text.
     * @see #setMessageText(String)
     */
    public String getMessageText() {
        return this.messageText;
    }

    /**
     * Set new value of hint (tool tip, status bar etc).
     *
     * @param pHint New value of hint.
     * @see #getHint()
     */
    public void setHint(final String pHint) {
        this.hint = pHint;
    }

    /**
     * Returns the hint.
     *
     * @return The hint.
     * @see #setHint(String)
     */
    public String getHint() {
        return this.hint;
    }

    /**
     * Stores an exception.
     *
     * @param exc Last exception thrown.
     * @see #getException()
     */
    public void setException(final Throwable exc) {
        this.thrown = exc;
    }

    /**
     * Returns the stored exception (if any).
     *
     * @return The stored exception (if any).
     * @see #setException(Throwable)
     */
    public Throwable getException() {
        return this.thrown;
    }

    /**
     * @return The current message context.
     */
    public MessageContext getContext() {
        return context;
    }

    /**
     * Set the new message context.
     *
     * @param context The new context.
     */
    public void setContext(MessageContext context) {
        this.context = context;
    }

    /**
     * Sets the new string reply. Ensures it is never null.
     *
     * @param newRepl New reply string.
     * @see #getReplString()
     */
    @SuppressWarnings("nls")
	public void setReplString(final String newRepl) {
        this.replString = (newRepl == null) ? "" : newRepl;
    }

    /**
     * Returns the reply String.
     *
     * @return The reply string.
     * @see #setReplString(String)
     */
    public String getReplString() {
        return this.replString;
    }

    /**
     * Getter for the property replyFile.
     *
     * @return The value of replyFile.
     * @see #setReplyFile(File)
     */
    public File getReplyFile() {
        return this.replyFile;
    }

    /**
     * Sets the new value for replyFile.
     *
     * @param replyFile The new value for replyFile.
     * @see #getReplyFile()
     */
    public void setReplyFile(File replyFile) {
        this.replyFile= replyFile;
    }

	/**
	 * Setter for the severity property.
	 * @param severity The severity to set
	 */
	public void setSeverity(Severity severity) {
		this.severity = severity;
	}

	/**
	 * Getter for the severity property.
	 * @return The severity
	 */
	public Severity getSeverity() {
		return severity;
	}

	/**
	 * Gets the messages contained by this message.
	 * @return The list of messages
	 */
	public List<Message> getMessages() {
		return messages;
	}

}
