/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.util.List;

import no.esito.util.BeanID;

import org.springframework.beans.factory.annotation.Autowired;

/**
 * This class is a singleton used to create messages.
 */
@BeanID("messageFactory")
public class MessageFactory {

    private List<MessageConfigurator> configurators;

    /** The list of messageConfigurators.
     *
     * @param configurators the list of configurators.
     */
    @Autowired(required = false)
    public void setConfigurators(List<MessageConfigurator> configurators) {
        this.configurators = configurators;
    }

    /**
     * This method should not be considered a part of the public api.
     *
     * @return messageConfigurators
     */
    List<MessageConfigurator> getConfigurators() {
        return configurators;
    }

    /**
     * Produces a message. The configurators are used to provide the message
     * text.
     *
     * @param id
     *            The message id.
     * @param args
     *            Used to replace keywords in the message text.
     * @return A new configured message.
     */
    public Message getMessage(String id, Object... args) {
        Message theMessage = new Message(id, args);
        if (getConfigurators() == null) {
            return theMessage;
        }
        for (MessageConfigurator messageConfigurator : getConfigurators()) {

            if (messageConfigurator.configure(theMessage)) {
                break; // finished configuring the message.
            }
        }
        return theMessage;
    }

    /**
     * @return A message factory used to produce messages.
     */
    @Deprecated
    public static MessageFactory getInstance() {
        return MessageSystem.getMessageFactory();
    }

}