/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.service.G9Spring;

/**
 * This class provides the static entry points to the message system.
 */
public final class MessageSystem {
    private static final Logger log = Logger.getLogger(MessageSystem.class);

	private MessageSystem(){
	    // EMPTY
	}

    /**
     * Used to indicate to the message dispatcher that there should be no interaction with this message.
     */
    public static final DispatcherContext NO_INTERACTION = new DispatcherContext() { /* EMPTY */ };

    /**
     * Get the message dispatcher.
     * 
     * @param dispatcherContext the dispatcher context
     * @return a message dispatcher with the specified dispatcher context
     */
    public static MessageDispatcher getMessageDispatcher(DispatcherContext dispatcherContext) {
        MessageDispatcherSetup dispatcher = ServiceLoader.getService(MessageDispatcherSetup.class);
        dispatcher.setDispatcherContext(dispatcherContext);
        return dispatcher;
    }

    /**
     * Get the message dispatcher, without a dispatcher context.
     * The dispatcher must provide a context when needed.
     * 
     * @return a message dispatcher with the specified dispatcher context
     */
    public static MessageDispatcher getMessageDispatcher() {
    	MessageDispatcherSetup dispatcher = ServiceLoader.getService(MessageDispatcherSetup.class);
    	return dispatcher;
    }
    
    /**
     * @return a message factory used to produce messages
     */
    public static MessageFactory getMessageFactory() {
        MessageFactory factoryInstance = G9Spring.getBean("messageFactory");
        if (factoryInstance == null) {
            log.warn("messageFactory is null - this probably means that this class has not bean loaded by Spring");
            log.warn("Creating a local (non-spring) instance");
            factoryInstance = new MessageFactory();
        }
        if (log.isTraceEnabled()) {
            log.trace("Returning message factory: " + factoryInstance.getClass().getSimpleName());
        }
        return factoryInstance;
    }
}
