/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message.metric;

/**
 * This is the interface that a timer should support.
 * A timer has certain characteristics:
 *   - The "key" - used to distinguish each timer in
 *     a Set.
 *   - The count - some timers will also count how many
 *     times a certain event has occurred.
 *   - The value - some timers will add upp numbers,
 *     often used in connection with the count to get
 *     an average.
 *   - And of course the time itself.
 *
 */
public interface IMetric extends Cloneable {


    /**
     * Add a value to a metric and increase the counter.
     * @param val    Value to add
     *
     */
    void add(long val);

    /**
     * Reset the metric.
     */
    void clear();

    /**
     * Make a clone of this object.
     *
     * @return a deep copy of this object
     * @throws CloneNotSupportedException if it fails...
     */
    Object clone() throws CloneNotSupportedException;

    /**
     * Returns the identifier.
     * @return key value
     * @see #setKey
     */
    String getKey();

    /**
     * Gets the count of updates to this metric.
     * @return count uf updates
     * @see #setCount
     */
    long getCount();

    /**
     * Returns the running time so far.
     * @return time in millis
     */
    long getTimer();

    /**
     * Sets the end time and returns a reference to this object.
     *
     * @return this object
     */
    IMetric stopTimer();

    /**
     * Gets the total value for this metric.
     * @return total value
     * @see #setValue
     */
    long getValue();

    /**
     * Sets the ID.
     * @param key ID
     * @see #getKey
     */
    void setKey(final String key);

    /**
     * Sets the counter value.
     * @param n    New counter value
     * @see #getCount
     */
    void setCount(final long n);

    /**
     * Sets the new value without changing the count.
     * @param n    New value
     * @see #getValue
     */
    void setValue(final long n);
}
