/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.os;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import no.esito.log.Logger;
import no.g9.support.Visitor;

/**
 * Representation of a node in the object selection.
 * 
 * @param <T> the domain class of the object selection node.
 */
public abstract class AbstractOSRole<T> implements OSRole<T> {

    /** The logger */
    private static final Logger log = Logger.getLogger(AbstractOSRole.class);

    /** The relation cardinality of the node. */
    public final RelationCardinality CARDINALITY;

    /** The relation type. */
    public final RelationType RELATION_TYPE;

    /** The parent of this node. */
    private final OSRole<?> parentNode;

    /** The role constant */
    private final RoleConstant roleConstant;

    /**
     * The list of this nodes children.
     */
    private final List<OSRole<?>> children = new ArrayList<OSRole<?>>();

    /** The list of keys. First element is always main key. */
    protected List<Key> keys;

    /**
     * Constructs a new object selection role.
     * 
     * @param parent the parent of this node.
     * @param relationType the type of relation to parent.
     * @param cardinality the cardinality of this node.
     * @param roleConstant the constant denoting this role.
     */
    protected AbstractOSRole(OSRole<?> parent, RelationType relationType,
            RelationCardinality cardinality, RoleConstant roleConstant) {
        this.parentNode = parent;
        RELATION_TYPE = relationType;
        CARDINALITY = cardinality;
        this.roleConstant = roleConstant;
        if (parent != null) {
            parent.addChild(this);
        }
    }

    @Override
    public RelationType getRelationType() {
        return RELATION_TYPE;
    }

    @Override
    public RelationCardinality getRelationCardinality() {
        return CARDINALITY;
    }

    @Override
    public Map<AttributeConstant, Object> getAttributeValues(
            Object domainInstance) {
        Map<AttributeConstant, Object> values = new HashMap<AttributeConstant, Object>();

        AttributeConstant[] attributes = getAttributeConstants();
        for (AttributeConstant attribute : attributes) {
            values.put(attribute, getValue(domainInstance, attribute));
        }

        return values;

    }

    /**
     * Gets the parent of this node.
     * 
     * @return the parent
     */
    @Override
    public OSRole<?> getParent() {
        return parentNode;
    }

    /**
     * Gets the role name of this node.
     * 
     * @return the roleName
     */
    @Override
    public final RoleConstant getRoleConstant() {
        return roleConstant;
    }

    /**
     * Returns a map of attributes values. The map key is the attribute name and
     * the value is the attribute value.
     * 
     * @param instance the domain instance
     * @param attributes a string array of attribute names
     * @return a map of attribute names and values
     */
    @Override
    public Map<AttributeConstant, Object> getValues(Object instance,
            AttributeConstant[] attributes) {
        Map<AttributeConstant, Object> values = new HashMap<AttributeConstant, Object>();
        for (AttributeConstant attribute : attributes) {
            values.put(attribute, getValue(instance, attribute));
        }
        return values;
    }

    /**
     * Adds a child to this (parent) node.
     * 
     * @param child the child to add
     * @throws IllegalArgumentException if this is not the parent of the child
     *             or the child is already added.
     */
    @Override
    public void addChild(OSRole<?> child) {
        if (child.getParent() != this) {
            throw new IllegalArgumentException(
                    "Can't add child that does not have this as parent.");
        }

        if (children.contains(child)) {
            throw new IllegalArgumentException("Child already added.");
        }

        children.add(child);
    }

    /**
     * Gets an unmodifiable list of child nodes.
     * 
     * @return the list of child nodes.
     */
    @Override
    public List<OSRole<?>> getChildren() {
        return Collections.unmodifiableList(children);
    }

    /**
     * Gets the child role for the given child role constant.
     * 
     * @param childRole the child role constant
     * @return the child role, or null if not found
     */
    @Override
    public OSRole<?> getChild(RoleConstant childRole) {
        for (OSRole<?> role : children) {
            if (role.getRoleConstant() == childRole) {
                return role;
            }
        }
        return null;
    }

    /**
     * If possible, returns a downcasted reference to the supplied object. The
     * reference is of type T.
     * 
     * @param obj the object to downcast
     * @return a casted version of the supplied object, or <code>null</code> if
     *         the object is not assignable to type T.
     */
    @Override
    public T castToType(Object obj) {
        if (getDomainClass().isInstance(obj)) {
            return getDomainClass().cast(obj);
        }
        return null;
    }

    /**
     * Returns a string with the role name prefixed with "Role: ".
     */
    @Override
    public String toString() {
        return String.valueOf(getRoleConstant());
    }

    /**
     * @param mainKey the mainKey to set
     */
    @Override
    public void setMainKey(Key mainKey) {
        if (keys == null) {
            keys = new ArrayList<Key>();
        } else if (keys.contains(mainKey)) {
            keys.remove(mainKey);
        }
        keys.add(0, mainKey);
    }

    /**
     * Gets the main key of the object selection role
     * 
     * @return the main key
     */
    @Override
    public Key getMainKey() {
        return getKeys().isEmpty() ? null : getKeys().get(0);
    }

    @Override
    public boolean equalsUsingKey(Key keyToUse, Object anObject,
            Object anOtherObject) {

        if (log.isTraceEnabled()) {
            log.trace(this + " equals using key " + keyToUse);
        }

        if (anObject == anOtherObject) {
            return true;
        }

        if (anObject == null) {
            return false;
        }

        boolean foundMatch = true;
        AttributeConstant[] attributes = keyToUse.getAttributes();
        for (int i = 0; i < attributes.length && foundMatch; i++) {
            Object value1 = getValue(anObject, attributes[i]);
            Object value2 = getValue(anOtherObject, attributes[i]);
            foundMatch = KeyTool.isDefined(value1) && KeyTool.isDefined(value2);
            // KeyTool also checks for null reference.
            if (foundMatch) {
                foundMatch = value1.equals(value2);
            }
            if (log.isTraceEnabled()) {
                log.trace("Comparing values in " + attributes[i] + ", value1: "
                        + value1 + ", value2: " + value2 + ", equals so far: "
                        + foundMatch);
            }
        }
        return foundMatch;

    }

    /**
     * Get an attribute value from a related role. A navigation path from the
     * given domain object to the attribute must be present in the object
     * selection.
     * 
     * @param domainObject the current domain object.
     * @param attribute the attribute value to get.
     * @return the attribute value, or null if not found.
     */
    protected Object getRelatedValue(Object domainObject,
            AttributeConstant attribute) {
        if (domainObject == null || attribute == null) {
            return null;
        }
        RoleConstant roleConst = getNextRelatedRole(attribute);
        OSRole<?> child = getChild(roleConst);
        if (child == null) {
            throw new IllegalArgumentException("Unknown related attribute: "
                    + attribute + " for " + this);
        }
        return child.getValue(getRelation(domainObject, roleConst), attribute);
    }

    /**
     * Set an attribute value on a related role. A navigation path from the
     * given domain object to the attribute must be present in the object
     * selection. If a related object is missing, a default empty domain object
     * is created.
     * 
     * @param domainObject the current domain object.
     * @param attribute the attribute value to set.
     * @param value the new attribute value.
     */
    protected void setRelatedValue(Object domainObject,
            AttributeConstant attribute, Object value) {

        if (log.isTraceEnabled()) {
            log.trace("Setting related value " + attribute + " = " + value);
        }
        
        if (domainObject == null || attribute == null) {
            return;
        }

        RoleConstant roleConst = getNextRelatedRole(attribute);
        OSRole<?> child = getChild(roleConst);
        if (child == null) {
            throw new IllegalArgumentException("Unknown related attribute: "
                    + attribute + " for " + this);
        }
        Object relatedObject = getRelation(domainObject, roleConst);
        if (relatedObject == null) {
            relatedObject = child.createNewInstance();
            setRelation(domainObject, relatedObject, roleConst);
            if (child.isNavigableToParent()) {
                child.setRelation(relatedObject, domainObject, this
                        .getRoleConstant());
            }
        }
        child.setValue(relatedObject, attribute, value);
    }

    private RoleConstant getNextRelatedRole(AttributeConstant attribute) {
        List<RoleConstant> pathRoles = attribute.getAttributeRole().getRolePath();
        int ix = pathRoles.indexOf(this.getRoleConstant());
        if (pathRoles.isEmpty() || ix == pathRoles.size() - 1) {
            return null; // No roles, or the current role is the last
        }
        return pathRoles.get(ix + 1);
    }

    @Override
    public boolean isRelated(AttributeConstant attribute) {
        return isAncestorOf(attribute.getAttributeRole());
    }
    
    @Override
    public boolean isAncestorOf(RoleConstant possibleHeir) {
        if (possibleHeir == this.getRoleConstant()) {
            return false;
        }

        return possibleHeir(possibleHeir);
        
    }

    
    
    private boolean possibleHeir(RoleConstant possibleHeir) {
        if (getRoleConstant() == possibleHeir) {
            return true;
        }

        boolean found = false;
        Iterator<OSRole<?>> childIterator = children.iterator();
        while (!found && childIterator.hasNext()) {
            AbstractOSRole<?> child = (AbstractOSRole<?>) childIterator.next();
            found = child.possibleHeir(possibleHeir);
        }
        
        return found;
        
    }

    @Override
    public boolean isMany() {
        return getRelationCardinality().equals(RelationCardinality.MANY);
    }

    @Override
    public boolean isRoot() {
        return getParent() == null;
    }
    
    @Override
    public OSRole<?> getRoot() {
        OSRole<?> root = this;
        while (!root.isRoot()) {
            root = root.getParent();
        }
        return root;
    }

    @Override
    public void accept(Visitor<OSRole<?>> visitor) {
        visitor.visit(this);
    }

    @Override
    public void visitBranch(Visitor<OSRole<?>> visitor) {
        accept(visitor);
        List<OSRole<?>> children2 = getChildren();
        for (OSRole<?> role : children2) {
            role.visitBranch(visitor);
        }
    }
}
