/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.os;

import java.util.Iterator;
import java.util.List;

import no.g9.os.AttributeConstant;
import no.g9.os.Key;
import no.g9.os.OSRole;

/**
 * The KeyTool class is a utility class, helping with various aspects of keys.
 * The class contains an <em>UNDEFINED_VALUE</em> value for all primitive java
 * types (e.g <code>UNDEFINED_VALUE_BYTE</code>, and methods to test if an
 * attribute is indeed defined.
 */
public class KeyTool {

    /** The default UNDEFINED value for a byte primitive (which is 0). */
    public static final byte UNDEFINED_VALUE_BYTE = 0;

    /** The default UNDEFINED value for a char primitive (which is 0). */
    public static final char UNDEFINED_VALUE_CHAR = 0;

    /** The default UNDEFINED value for an integer primitive (which is 0). */
    public static final int UNDEFINED_VALUE_INT = 0;

    /** The default UNDEFINED value for a long primitive (which is 0). */
    public static final long UNDEFINED_VALUE_LONG = 0;

    /** The default UNDEFINED value for a float primitive (which is 0.0). */
    public static final float UNDEFINED_VALUE_FLOAT = 0;

    /** The default UNDEFINED value for a double primitive (which is 0.0). */
    public static final double UNDEFINED_VALUE_DOUBLE = 0;

    /** The default UNDEFINED value for a boolean primitive (which is false). */
    public static final boolean UNDEFINED_VALUE_BOOLEAN = false;

    /** The default UNDEFINED value for a String (which is an empty String). */
    public static final String UNDEFINED_VALUE_STRING = "";

    /**
     * Searches the specified instance for a defined key (a key where all
     * attributes are defined), and returns the name of the first completely
     * defined key.
     * 
     * @param instance the instance.
     * @param role the instance role.
     * @return the complete unique key.
     */
    public static Key getCompleteUniqueKey(Object instance, OSRole<?> role) {
        List<Key> keys = role.getKeys();
        Iterator<Key> keyIterator = keys.iterator();
        Key found = null;

        while (found == null && keyIterator.hasNext()) {
            Key key = keyIterator.next();
            AttributeConstant[] attributes = key.getAttributes();

            if (attributesDefined(instance, role, attributes)) {
                found = key;
            }

        }

        return found;
    }

    /**
     * Check if the specified instance contains defined values for all the
     * attributes specified attribute array (see various <code>isDefined</code>
     * methods).
     * 
     * @param instance the instance to check attributes in.
     * @param role the object selection role of the instance.
     * @param attributes an array containing the attribute names.
     * @return <code>true</code> if all values defined by the attribute array
     *         are defined.
     */
    public static boolean attributesDefined(Object instance, OSRole<?> role,
            AttributeConstant[] attributes) {
        boolean isDefined = true;

        for (int i = 0; i < attributes.length && isDefined; i++) {
            isDefined = isDefined(role.getValue(instance, attributes[i]));
        }

        return isDefined;
    }

    /**
     * Check if the specified primitive is defined. A primitive value is defined
     * if it's value is different from the primitives declared UNDEFINED value.
     * 
     * @param primitive the value to verify
     * @return <code>false</code> if the specified value equals the primitive's
     *         UNDEFINED value, otherwise <code>true</code>
     * @see #UNDEFINED_VALUE_BYTE
     */
    public static boolean isDefined(byte primitive) {
        return primitive != UNDEFINED_VALUE_BYTE;
    }

    /**
     * Check if the specified primitive wrapper is defined. A primitive wrapper
     * is defined if it is not <code>null</code> and the wrapped primitive value
     * is defined.
     * 
     * @param b the wrapper to verify.
     * @return <code>false</code> if the wrapper is either <code>null</code> or
     *         wrapped primitive equals the primitive's UNDEFINED value.
     */
    public static boolean isDefined(Byte b) {
        return b != null ? isDefined(b.byteValue()) : false;
    }

    /**
     * Check if the specified primitive is defined. A primitive value is defined
     * if it's value is different from the primitives declared UNDEFINED value.
     * 
     * @param primitive the value to verify
     * @return <code>false</code> if the specified value equals the primitive's
     *         UNDEFINED value, otherwise <code>true</code>.
     * @see #UNDEFINED_VALUE_CHAR
     */
    public static boolean isDefined(char primitive) {
        return primitive != UNDEFINED_VALUE_CHAR;
    }

    /**
     * Check if the specified primitive wrapper is defined. A primitive wrapper
     * is defined if it is not <code>null</code> and the wrapped primitive value
     * is defined.
     * 
     * @param c the wrapper to verify.
     * @return <code>false</code> if the wrapper is either <code>null</code> or
     *         wrapped primitive equals the primitive's UNDEFINED value.
     * @see #isDefined(char)
     */
    public static boolean isDefined(Character c) {
        return c != null ? isDefined(c.charValue()) : false;
    }

    /**
     * Check if the specified primitive is defined. A primitive value is defined
     * if it's value is different from the primitives declared UNDEFINED value.
     * 
     * @param primitive the value to verify
     * @return <code>false</code> if the specified value equals the primitive's
     *         UNDEFINED value, otherwise <code>true</code>.
     * @see #UNDEFINED_VALUE_INT
     */
    public static boolean isDefined(int primitive) {
        return primitive != UNDEFINED_VALUE_INT;
    }

    /**
     * Check if the specified primitive wrapper is defined. A primitive wrapper
     * is defined if it is not <code>null</code> and the wrapped primitive value
     * is defined.
     * 
     * @param i the wrapper to verify.
     * @return <code>false</code> if the wrapper is either <code>null</code> or
     *         wrapped primitive equals the primitive's UNDEFINED value.
     * @see #isDefined(int)
     */
    public static boolean isDefined(Integer i) {
        return i != null ? isDefined(i.intValue()) : false;
    }

    /**
     * Check if the specified primitive is defined. A primitive value is defined
     * if it's value is different from the primitives declared UNDEFINED value.
     * 
     * @param primitive the value to verify
     * @return <code>false</code> if the specified value equals the primitive's
     *         UNDEFINED value, otherwise <code>true</code>.
     * @see #UNDEFINED_VALUE_LONG
     */
    public static boolean isDefined(long primitive) {
        return primitive != UNDEFINED_VALUE_LONG;
    }

    /**
     * Check if the specified primitive wrapper is defined. A primitive wrapper
     * is defined if it is not <code>null</code> and the wrapped primitive value
     * is defined.
     * 
     * @param l the wrapper to verify.
     * @return <code>false</code> if the wrapper is either <code>null</code> or
     *         wrapped primitive equals the primitive's UNDEFINED value.
     * @see #isDefined(long)
     */
    public static boolean isDefined(Long l) {
        return l != null ? isDefined(l.longValue()) : false;
    }

    /**
     * Check if the specified primitive is defined. A primitive value is defined
     * if it's value is different from the primitives declared UNDEFINED value.
     * 
     * @param primitive the value to verify
     * @return <code>false</code> if the specified value equals the primitive's
     *         UNDEFINED value, otherwise <code>true</code>.
     * @see #UNDEFINED_VALUE_FLOAT
     */
    public static boolean isDefined(float primitive) {
        return primitive != UNDEFINED_VALUE_FLOAT;
    }

    /**
     * Check if the specified primitive wrapper is defined. A primitive wrapper
     * is defined if it is not <code>null</code> and the wrapped primitive value
     * is defined.
     * 
     * @param f the wrapper to verify.
     * @return <code>false</code> if the wrapper is either <code>null</code> or
     *         wrapped primitive equals the primitive's UNDEFINED value.
     * @see #isDefined(float)
     */
    public static boolean isDefined(Float f) {
        return f != null ? isDefined(f.floatValue()) : false;
    }

    /**
     * Check if the specified primitive is defined. A primitive value is defined
     * if it's value is different from the primitives declared UNDEFINED value.
     * 
     * @param primitive the value to verify
     * @return <code>false</code> if the specified value equals the primitive's
     *         UNDEFINED value, otherwise <code>true</code>.
     * @see #UNDEFINED_VALUE_DOUBLE
     */
    public static boolean isDefined(double primitive) {
        return primitive != UNDEFINED_VALUE_DOUBLE;
    }

    /**
     * Check if the specified primitive wrapper is defined. A primitive wrapper
     * is defined if it is not <code>null</code> and the wrapped primitive value
     * is defined.
     * 
     * @param d the wrapper to verify.
     * @return <code>false</code> if the wrapper is either <code>null</code> or
     *         wrapped primitive equals the primitive's UNDEFINED value.
     * @see #isDefined(double)
     */
    public static boolean isDefined(Double d) {
        return d != null ? isDefined(d.doubleValue()) : false;
    }

    /**
     * Check if the specified primitive is defined. A primitive value is defined
     * if it's value is different from the primitives declared UNDEFINED value.
     * 
     * @param primitive the value to verify
     * @return <code>false</code> if the specified value equals the primitive's
     *         UNDEFINED value, otherwise <code>true</code>.
     * @see #UNDEFINED_VALUE_BOOLEAN
     */
    public static boolean isDefined(boolean primitive) {
        return primitive != UNDEFINED_VALUE_BOOLEAN;
    }

    /**
     * Check if the specified primitive wrapper is defined. A primitive wrapper
     * is defined if it is not <code>null</code> and the wrapped primitive value
     * is defined.
     * 
     * @param b the wrapper to verify.
     * @return <code>false</code> if the wrapper is either <code>null</code> or
     *         wrapped primitive equals the primitive's UNDEFINED value.
     */
    public static boolean isDefined(Boolean b) {
        return b != null ? isDefined(b.booleanValue()) : false;
    }

    /**
     * Check if the specified String is defined. A String is defined if it
     * contains any characters (including whitespace).
     * 
     * @param s the String to verify
     * @return <code>false</code> if the specified String is null or of zero
     *         length.
     * @see #UNDEFINED_VALUE_STRING
     */
    public static boolean isDefined(String s) {
        return s != null && !UNDEFINED_VALUE_STRING.equals(s);
    }

    /**
     * Check if the specified object is defined. An object is defined if it's
     * not a <code>null</code> reference.
     * <p>
     * If the object is a primitive wrapper (Byte, Character, Integer, Long,
     * Float or Double), or a String, the reference is narrowed and the
     * corresponding isDefined method is invoked (e.g. if this method is invoked
     * with an Integer reference, the object is casted to an Integer, and the
     * <code>isDefined(Integer)</code> method is invoked).
     * 
     * @param o the value to verify
     * @return <code>false</code> if the specified value is undefined, otherwise
     *         <code>true</code>.
     */
    public static boolean isDefined(Object o) {

        if (o instanceof String) {
            return isDefined((String) o);
        }

        if (o instanceof Byte) {
            return isDefined((Byte) o);
        }

        if (o instanceof Character) {
            return isDefined((Character) o);
        }

        if (o instanceof Integer) {
            return isDefined((Integer) o);
        }

        if (o instanceof Long) {
            return isDefined((Long) o);
        }

        if (o instanceof Float) {
            return isDefined((Float) o);
        }

        if (o instanceof Double) {
            return isDefined((Double) o);
        }

        if (o instanceof Boolean) {
            return isDefined((Boolean) o);
        }

        return o != null;
    }
}
