/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.service.print;

import java.awt.print.Book;
import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;

import javax.print.PrintServiceLookup;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.print.JRPrinterAWT;
import no.g9.exception.G9ServiceException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.MessageDispatcher;
import no.g9.message.MessageSystem;

/**
 * The class extends JRPrinterAWT with printing to a named printer.
 */
public class JasperExplicitPrinterPrintManager extends JRPrinterAWT {

    /** The print (image) to be printed. */
    private JasperPrint jasperPrint;

    /**
     * Constructor.
     *
     * @param jrPrint       The JasperPrint
     * @throws JRException  Initialization may fail
     */
    protected JasperExplicitPrinterPrintManager(JasperPrint jrPrint) throws JRException {
        super(jrPrint);
        this.jasperPrint= jrPrint;
    }

    /**
     * Prints the given image to a given printer.
     *
     * @param jrPrint       The print image
     * @param printerName   The name of the printer to print on
     */
    public static void print(JasperPrint jrPrint, String printerName) {
        int lastPageIndex= jrPrint.getPages().size()-1;

        try {
            printPages(jrPrint, 0, lastPageIndex, printerName);
        }
        catch (JRException e) {
            throw new G9ServiceException(e);
        }
    }

    /**
     * Prints the given pages of the print..
     *
     * @param jrPrint (missing javadoc)
     * @param firstPageIndex (missing javadoc)
     * @param lastPageIndex (missing javadoc)
     * @param printerName (missing javadoc)
     * @throws JRException (missing javadoc)
     */
    public static void printPages(JasperPrint jrPrint,
                                     int firstPageIndex,
                                     int lastPageIndex,
                                     String printerName)
                            throws JRException {
        JasperExplicitPrinterPrintManager printer= new JasperExplicitPrinterPrintManager(jrPrint);
        printer.printPages(firstPageIndex, lastPageIndex, printerName);
    }

    /**
     * Prints the range of pages from first page to last page on the given printer.
     *
     * @param firstPageIndex    Index of the first page to be printed
     * @param lastPageIndex     Index of the last page to be printed
     * @param printerName       Name of the printer to print on
     * @throws G9ServiceException if the page index is out of range.
     */
    private void printPages(int firstPageIndex,
                              int lastPageIndex,
                              String printerName)
                                throws G9ServiceException {
       if (firstPageIndex < 0 || firstPageIndex > lastPageIndex
            || lastPageIndex >= jasperPrint.getPages().size()) {
           throw new G9ServiceException("Invalid page index range : "
                                  + firstPageIndex + " - " + lastPageIndex
                                  + " of " + jasperPrint.getPages().size());
       }
       PrinterJob printJob= PrinterJob.getPrinterJob();
        // fix for bug ID 6255588 from Sun bug database
       initPrinterJobFields(printJob);

       javax.print.PrintService[] pservices= PrintServiceLookup
                                .lookupPrintServices(null, null);
       boolean found = false;
       if (pservices.length > 0) {
           for (int i= 0; i < pservices.length; i++) {
               System.err.println(pservices[i].getName());
               String name= pservices[i].getName();
               if (printerName.equals(name)|| matchEnd(printerName, name)) {
                   try {
                       printJob.setPrintService(pservices[i]);
                       found = true;
                       break;
                   }
                   catch (PrinterException e) {
                        getMessageDispatcher().dispatch(
                                CRuntimeMsg.PM_SETTING_PRINTER_SERVICE_FAILED,
                                printerName);
                       throw new G9ServiceException(e);
                   }
               }
           }
       }
       if (!found) {
            getMessageDispatcher().dispatch(CRuntimeMsg.PM_NO_SUCH_PRINTER,
                    printerName);
       }

       PageFormat pageFormat= printJob.defaultPage();
       Paper paper= pageFormat.getPaper();
       printJob.setJobName("JasperReports - " + jasperPrint.getName());

       //inform printer of page layouts - replaced the deprecated methods and enums
       switch (jasperPrint.getOrientationValue()) {
       case  LANDSCAPE: {
           pageFormat.setOrientation(PageFormat.LANDSCAPE);
           paper.setSize(jasperPrint.getPageHeight(), jasperPrint
                                    .getPageWidth());
           paper.setImageableArea(0, 0, jasperPrint.getPageHeight(),
                                   jasperPrint.getPageWidth());
           break;
        }
        case PORTRAIT:
        default: {
            pageFormat.setOrientation(PageFormat.PORTRAIT);
            paper.setSize(jasperPrint.getPageWidth(), jasperPrint
                                    .getPageHeight());
            paper.setImageableArea(0, 0, jasperPrint.getPageWidth(),
                                   jasperPrint.getPageHeight());
        }
    }

        pageFormat.setPaper(paper);

        Book book= new Book();
        book.append(this, pageFormat, lastPageIndex - firstPageIndex + 1);
        printJob.setPageable(book);
        try {
            printJob.print();
        }
        catch (Exception ex) {
            throw new G9ServiceException("Error printing report.", ex);
        }
    }

    /**
     * The name without server and so forth.
     *
     * @param longName  Complete name
     * @return          Short name of the printer
     */
    private MessageDispatcher getMessageDispatcher() {
        return MessageSystem.getMessageDispatcher(null);
    }


    /**
     * @param lookupName name
     * @param pPrinterName printer name
     * @return .
     */
    // TODO: doc
    boolean matchEnd(String lookupName, String pPrinterName) {
        return pPrinterName.toLowerCase().endsWith(lookupName.toLowerCase());
    }


}
