/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.Serializable;

/**
 * Enumerator for the set of supported find methods.
 */
public final class EFindMethod implements Serializable {



	/**
     * The name of this find method.
     */
    protected final String name;

    /**
     * The default find method, using key properties.
     */
    public final static EFindMethod DEFAULT = new EFindMethod("default");

    /**
     * The query find method, using a query (possibly with query arguments).
     */
    public final static EFindMethod QUERY = new EFindMethod("query");

    /**
     * The example find method, using query by example.
     */
    public final static EFindMethod EXAMPLE = new EFindMethod("example");

    /**
     * The criteria find method, using a tree of criterions.
     */
    public final static EFindMethod CRITERIA = new EFindMethod("criteria");

    /**
     * The custom find method. The application programmer is responsible
     * for handling the find, by overriding a method in the generated service
     * manager class, or implementing a hook method.
     *
     * To transfer data to the find, use any of the available fields in
     * <code>FindData</code>, including the general <code>customData</code>.
     */
    public final static EFindMethod CUSTOM = new EFindMethod("custom");

    /** Hidden constructor
     * @param name Method identifier */
    private EFindMethod(String name) {
        this.name = name;
    }

    /**
     * @return the name of this find method.
     */
    public String name() {
        return this.name;
    }

    @Override
    public String toString() {
        return this.name();
    }

    @Override
    public boolean equals(final Object other) {
        if (this == other) {
            return true;
        }
        if (other instanceof EFindMethod ) {
        	 final EFindMethod that = (EFindMethod) other;
             return this.name.equals(that.name);
        }
        return false;
    }

    @Override
	public int hashCode() {
		final int prime = 31;
		return prime + ((name == null) ? 0 : name.hashCode());
	}
}
