/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.castor;

import org.exolab.castor.mapping.GeneralizedFieldHandler;
import java.time.LocalDateTime;

/**
 * Castor field handler used to convert to/from XML fields of type
 * <code>xs:LocalDateTime</code> for the Joda time LocalDateTime.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches). 
 * <p>
 * <strong>NOTE:</strong> This class is
 * referenced from Castor mapping files, so refactor with caution!
 */
public class LocalDateTimeFieldHandler extends GeneralizedFieldHandler {

	/**
	 * Creates a new LocalDateTimeFieldHandler instance
	 */
	public LocalDateTimeFieldHandler() {
		super();
	}

	/**
	 * This method is used to convert the value when the getValue method is
	 * called. The getValue method will obtain the actual field value from given
	 * 'parent' object. This convert method is then invoked with the field's
	 * value. The value returned from this method will be the actual value
	 * returned by getValue method.
	 * 
	 * @param value
	 *            the object value to convert after performing a get operation
	 * @return the converted value.
	 */
	@Override
    public Object convertUponGet(Object value) {
		if (value == null)
			return null;
		LocalDateTime date = (LocalDateTime) value;
		return date.toString();
	}

	/**
	 * This method is used to convert the value when the setValue method is
	 * called. The setValue method will call this method to obtain the converted
	 * value. The converted value will then be used as the value to set for the
	 * field.
	 * 
	 * @param value
	 *            the object value to convert before performing a set operation
	 * @return the converted value.
	 */
	@Override
    public Object convertUponSet(Object value) {
		return LocalDateTime.parse((String)value);
	}

	/**
	 * Returns the class type for the field that this GeneralizedFieldHandler
	 * converts to and from. This should be the type that is used in the object
	 * model.
	 * 
	 * @return the class type of of the field
	 */
	@Override
    public Class<LocalDateTime> getFieldType() {
		return LocalDateTime.class;
	}

	/**
	 * Creates a new instance of the object described by this field.
	 * 
	 * @param parent
	 *            The object for which the field is created
	 * @return A new instance of the field's value
	 * @throws IllegalStateException
	 *             This field is a simple type and cannot be instantiated
	 */
	@Override
    public Object newInstance(Object parent) throws IllegalStateException {
		// -- Since it's marked as a string, just return null,
		// -- it's not needed.
		return null;
	}
}
