/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.transport;

import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.converter.ConverterFactory;

/**
 * A factory for conversion of Java enums to Java enums.
 * 
 * <strong>WARNING:</strong> Although this class is public, it should not be treated as part of the public API, as it
 * might change in incompatible ways between releases (even patches).
 */
@SuppressWarnings("rawtypes")
public class JavaEnumToJavaEnumConverterFactory implements ConverterFactory<Enum, Enum> {

    /*
     * (non-Javadoc)
     * 
     * @see org.springframework.core.convert.converter.ConverterFactory#getConverter (java.lang.Class)
     */
    @Override
    public <T extends Enum> Converter<Enum, T> getConverter(Class<T> targetType) {
        return new JavaEnumToJavaEnumConverter<T>(targetType);
    }

    /**
     * A converter for java enums to java enums.
     * 
     * @param <T>
     *            the generic type
     */
    private final static class JavaEnumToJavaEnumConverter<T extends Enum> implements Converter<Enum, T> {

        /** The enum type. */
        private Class<T> enumType;

        /**
         * Instantiates a new java enum to java enum converter.
         * 
         * @param enumType
         *            the enum type
         */
        public JavaEnumToJavaEnumConverter(Class<T> enumType) {
            this.enumType = enumType;

        }

        /*
         * (non-Javadoc)
         * 
         * @see org.springframework.core.convert.converter.Converter#convert(java .lang.Object)
         */
        @Override
        @SuppressWarnings("unchecked")
        public T convert(Enum source) {
            return (T) Enum.valueOf(enumType, source.name());

        }

    }

}
